<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Department;
use App\Models\User;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\IncomingOperation;
use App\Models\Warehouse\IncomingOperationItem;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\LpoManagement;
use App\Models\Warehouse\QualityInspectionChecklist;
use App\Models\Warehouse\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class IncomingOperationController extends Controller
{
    /**
     * Display a listing of incoming operations
     */
    public function index(Request $request)
    {
        $query = IncomingOperation::with(['supplier', 'project', 'division', 'receiver', 'inspector'])
                                 ->latest('operation_date');

        // Apply filters
        if ($request->filled('operation_type')) {
            $query->where('operation_type', $request->operation_type);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('division_id')) {
            $query->where('division_id', $request->division_id);
        }

        if ($request->filled('date_from')) {
            $query->where('operation_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('operation_date', '<=', $request->date_to);
        }

        $operations = $query->paginate(15);

        // Get filter options
        $suppliers = Supplier::where('status', 'active')->get();
        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();

        return view('warehouse.incoming-operations.index', compact(
            'operations', 'suppliers', 'projects', 'divisions'
        ));
    }

    /**
     * Show the form for creating a new incoming operation
     */
    public function create(Request $request)
    {
        $operationType = $request->get('type', 'supplier_delivery');

        $suppliers = Supplier::where('status', 'active')->get();
        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();
        $inspectors = User::whereHas('role', function ($query) {
            $query->whereIn('name', ['warehouse_inspector', 'warehouse_manager']);
        })->get();
        $lpos = LpoManagement::where('status', 'approved')->get();

        // Get currency exchange rates (this could be from a service)
        $exchangeRates = [
            'AED' => 1.0000,
            'USD' => 3.6725, // Example rate
            'EUR' => 4.0250  // Example rate
        ];

        return view('warehouse.incoming-operations.create', compact(
            'operationType', 'suppliers', 'projects', 'divisions',
            'inspectors', 'lpos', 'exchangeRates'
        ));
    }

    /**
     * Store a newly created incoming operation
     */
    public function store(Request $request)
    {
        $request->validate([
            'operation_type' => 'required|in:supplier_delivery,site_return,emergency_procurement',
            'operation_date' => 'required|date',
            'division_id' => 'required|exists:departments,id',
            'received_by' => 'required|exists:users,id',

            // Conditional validations
            'supplier_id' => 'required_if:operation_type,supplier_delivery|exists:suppliers,id',
            'project_id' => 'nullable|exists:projects,id',
            'lpo_number' => 'nullable|string',
            'delivery_note_number' => 'nullable|string',
            'currency' => 'required|in:AED,USD,EUR',
            'exchange_rate' => 'required|numeric|min:0.0001',

            // Items
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'required|exists:items,id',
            'items.*.quantity_delivered' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',

            // Files
            'delivery_note_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240',
            'photos.*' => 'nullable|file|mimes:jpg,jpeg,png|max:5120'
        ]);

        DB::beginTransaction();

        try {
            // Create the operation
            $operation = IncomingOperation::create([
                'operation_type' => $request->operation_type,
                'operation_date' => $request->operation_date,
                'status' => 'draft',
                'supplier_id' => $request->supplier_id,
                'project_id' => $request->project_id,
                'project_phase' => $request->project_phase,
                'division_id' => $request->division_id,
                'site_location' => $request->site_location,
                'gps_coordinates' => $request->gps_coordinates ? json_decode($request->gps_coordinates) : null,
                'lpo_number' => $request->lpo_number,
                'currency' => $request->currency,
                'exchange_rate' => $request->exchange_rate,
                'delivery_note_number' => $request->delivery_note_number,
                'vehicle_number' => $request->vehicle_number,
                'driver_name' => $request->driver_name,
                'driver_contact' => $request->driver_contact,
                'delivery_datetime' => $request->delivery_datetime,
                'delivery_instructions' => $request->delivery_instructions,
                'requires_inspection' => $request->boolean('requires_inspection', true),
                'inspector_id' => $request->inspector_id,
                'vat_rate' => 5.00, // UAE VAT rate
                'received_by' => $request->received_by,
                'received_by_name' => $request->received_by_name,
                'notes' => $request->notes
            ]);

            // Handle delivery note file upload
            if ($request->hasFile('delivery_note_file')) {
                $path = $request->file('delivery_note_file')->store(
                    'delivery-notes/' . $operation->id, 'public'
                );
                $operation->delivery_note_path = $path;
            }

            // Handle photo documentation
            $photoPaths = [];
            if ($request->hasFile('photos')) {
                foreach ($request->file('photos') as $photo) {
                    $photoPaths[] = $photo->store(
                        'incoming-operations/' . $operation->id . '/photos', 'public'
                    );
                }
                $operation->photo_documentation = $photoPaths;
            }

            // Create operation items
            $totalAmount = 0;
            foreach ($request->items as $itemData) {
                $unitPriceAed = $itemData['unit_price'];
                if ($request->currency !== 'AED') {
                    $unitPriceAed = $itemData['unit_price'] * $request->exchange_rate;
                }

                $item = IncomingOperationItem::create([
                    'incoming_operation_id' => $operation->id,
                    'item_id' => $itemData['item_id'],
                    'quantity_delivered' => $itemData['quantity_delivered'],
                    'unit_price' => $itemData['unit_price'],
                    'currency' => $request->currency,
                    'unit_price_aed' => $unitPriceAed,
                    'batch_number' => $itemData['batch_number'] ?? null,
                    'serial_number' => $itemData['serial_number'] ?? null,
                    'manufacturing_date' => $itemData['manufacturing_date'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                    'manufacturer' => $itemData['manufacturer'] ?? null,
                    'country_of_origin' => $itemData['country_of_origin'] ?? null,
                    'storage_location' => $itemData['storage_location'] ?? null,
                    'storage_conditions' => $itemData['storage_conditions'] ?? null,
                    'hazardous_material' => $itemData['hazardous_material'] ?? false,
                    'warranty_period_months' => $itemData['warranty_period_months'] ?? null,
                    'allocated_project_id' => $itemData['allocated_project_id'] ?? $request->project_id,
                    'project_phase' => $itemData['project_phase'] ?? $request->project_phase,
                    'work_area' => $itemData['work_area'] ?? null
                ]);

                $totalAmount += $item->total_price_aed;
            }

            // Update operation total amount
            $operation->total_amount = $totalAmount / $request->exchange_rate;
            $operation->save();

            // Move to pending inspection if required
            if ($operation->requires_inspection) {
                $operation->status = 'pending_inspection';
                $operation->save();
            }

            DB::commit();

            return redirect()->route('warehouse.incoming-operations.show', $operation)
                           ->with('success', 'Incoming operation created successfully.');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                        ->with('error', 'Failed to create incoming operation: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified incoming operation
     */
    public function show(IncomingOperation $operation)
    {
        $operation->load([
            'supplier', 'project', 'division', 'receiver',
            'inspector', 'approver', 'items.item', 'lpo'
        ]);

        return view('warehouse.incoming-operations.show', compact('operation'));
    }

    /**
     * Show the form for editing the specified incoming operation
     */
    public function edit(IncomingOperation $operation)
    {
        if ($operation->status === 'completed') {
            return back()->with('error', 'Cannot edit completed operations.');
        }

        $operation->load(['items.item']);

        $suppliers = Supplier::where('status', 'active')->get();
        $projects = Project::all();
        $divisions = Department::where('is_active', true)->get();
        $inspectors = User::whereHas('role', function ($query) {
            $query->whereIn('name', ['warehouse_inspector', 'warehouse_manager']);
        })->get();
        $lpos = LpoManagement::where('status', 'approved')->get();

        $exchangeRates = [
            'AED' => 1.0000,
            'USD' => 3.6725,
            'EUR' => 4.0250
        ];

        return view('warehouse.incoming-operations.edit', compact(
            'operation', 'suppliers', 'projects', 'divisions',
            'inspectors', 'lpos', 'exchangeRates'
        ));
    }

    /**
     * Update the specified incoming operation
     */
    public function update(Request $request, IncomingOperation $operation)
    {
        if ($operation->status === 'completed') {
            return back()->with('error', 'Cannot update completed operations.');
        }

        $request->validate([
            'operation_date' => 'required|date',
            'division_id' => 'required|exists:departments,id',
            'supplier_id' => 'nullable|exists:suppliers,id',
            'project_id' => 'nullable|exists:projects,id',
            'currency' => 'required|in:AED,USD,EUR',
            'exchange_rate' => 'required|numeric|min:0.0001',
            'items' => 'required|array|min:1',
            'delivery_note_file' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:10240'
        ]);

        DB::beginTransaction();

        try {
            // Update operation
            $operation->update($request->only([
                'operation_date', 'division_id', 'supplier_id', 'project_id',
                'project_phase', 'site_location', 'lpo_number', 'currency',
                'exchange_rate', 'delivery_note_number', 'vehicle_number',
                'driver_name', 'driver_contact', 'delivery_datetime',
                'delivery_instructions', 'inspector_id', 'notes'
            ]));

            // Handle file upload
            if ($request->hasFile('delivery_note_file')) {
                // Delete old file
                if ($operation->delivery_note_path) {
                    Storage::disk('public')->delete($operation->delivery_note_path);
                }

                $path = $request->file('delivery_note_file')->store(
                    'delivery-notes/' . $operation->id, 'public'
                );
                $operation->delivery_note_path = $path;
                $operation->save();
            }

            // Update items
            $operation->items()->delete(); // Remove existing items
            $totalAmount = 0;

            foreach ($request->items as $itemData) {
                $unitPriceAed = $itemData['unit_price'];
                if ($request->currency !== 'AED') {
                    $unitPriceAed = $itemData['unit_price'] * $request->exchange_rate;
                }

                $item = IncomingOperationItem::create([
                    'incoming_operation_id' => $operation->id,
                    'item_id' => $itemData['item_id'],
                    'quantity_delivered' => $itemData['quantity_delivered'],
                    'unit_price' => $itemData['unit_price'],
                    'currency' => $request->currency,
                    'unit_price_aed' => $unitPriceAed,
                    'batch_number' => $itemData['batch_number'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                    'allocated_project_id' => $itemData['allocated_project_id'] ?? $request->project_id
                ]);

                $totalAmount += $item->total_price_aed;
            }

            // Update total amount
            $operation->total_amount = $totalAmount / $request->exchange_rate;
            $operation->save();

            DB::commit();

            return redirect()->route('warehouse.incoming-operations.show', $operation)
                           ->with('success', 'Incoming operation updated successfully.');

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()
                        ->with('error', 'Failed to update incoming operation: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified incoming operation
     */
    public function destroy(IncomingOperation $operation)
    {
        if ($operation->status === 'completed') {
            return back()->with('error', 'Cannot delete completed operations.');
        }

        try {
            // Delete associated files
            if ($operation->delivery_note_path) {
                Storage::disk('public')->delete($operation->delivery_note_path);
            }

            if ($operation->photo_documentation) {
                foreach ($operation->photo_documentation as $photoPath) {
                    Storage::disk('public')->delete($photoPath);
                }
            }

            $operation->delete();

            return redirect()->route('warehouse.incoming-operations.index')
                           ->with('success', 'Incoming operation deleted successfully.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to delete incoming operation: ' . $e->getMessage());
        }
    }

    /**
     * Perform quality inspection
     */
    public function inspect(Request $request, IncomingOperation $operation)
    {
        if ($operation->status !== 'pending_inspection') {
            return back()->with('error', 'Operation is not pending inspection.');
        }

        $request->validate([
            'inspection_result' => 'required|in:passed,failed,conditional',
            'inspection_notes' => 'nullable|string',
            'items' => 'required|array',
            'items.*.id' => 'required|exists:incoming_operation_items,id',
            'items.*.quality_status' => 'required|in:passed,failed,conditional',
            'items.*.quality_notes' => 'nullable|string'
        ]);

        DB::beginTransaction();

        try {
            // Update operation inspection
            $operation->update([
                'status' => $request->inspection_result === 'passed' ? 'approved' : 'rejected',
                'inspection_datetime' => now(),
                'inspection_result' => $request->inspection_result,
                'inspection_notes' => $request->inspection_notes
            ]);

            // Update item quality status
            foreach ($request->items as $itemData) {
                $item = IncomingOperationItem::find($itemData['id']);
                $item->update([
                    'quality_status' => $itemData['quality_status'],
                    'quality_notes' => $itemData['quality_notes'] ?? null
                ]);
            }

            DB::commit();

            $message = $request->inspection_result === 'passed' ?
                      'Inspection completed. Operation approved.' :
                      'Inspection completed. Operation rejected.';

            return redirect()->route('warehouse.incoming-operations.show', $operation)
                           ->with('success', $message);

        } catch (\Exception $e) {
            DB::rollback();
            return back()->with('error', 'Failed to complete inspection: ' . $e->getMessage());
        }
    }

    /**
     * Complete the operation and update inventory
     */
    public function complete(IncomingOperation $operation)
    {
        if ($operation->status !== 'approved') {
            return back()->with('error', 'Operation must be approved before completion.');
        }

        try {
            $operation->complete();

            return redirect()->route('warehouse.incoming-operations.show', $operation)
                           ->with('success', 'Operation completed successfully. Inventory updated.');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to complete operation: ' . $e->getMessage());
        }
    }

    /**
     * Print goods received note
     */
    public function print(IncomingOperation $operation)
    {
        $operation->load(['supplier', 'project', 'division', 'receiver', 'items.item']);

        return view('warehouse.incoming-operations.print', compact('operation'));
    }

    /**
     * Get items for AJAX requests
     */
    public function getItems(Request $request)
    {
        $search = $request->get('search');

        $items = Item::with('category')
                    ->where('status', 'active')
                    ->when($search, function ($query, $search) {
                        return $query->where('item_code', 'like', "%{$search}%")
                                   ->orWhere('item_description', 'like', "%{$search}%");
                    })
                    ->limit(20)
                    ->get();

        return response()->json($items);
    }

    /**
     * Get LPO details for AJAX requests
     */
    public function getLpoDetails(Request $request)
    {
        $lpoNumber = $request->get('lpo_number');

        $lpo = LpoManagement::with(['items.item', 'supplier', 'project'])
                           ->where('lpo_number', $lpoNumber)
                           ->first();

        if (!$lpo) {
            return response()->json(['error' => 'LPO not found'], 404);
        }

        return response()->json($lpo);
    }

    /**
     * Get quality checklists for material category
     */
    public function getQualityChecklists(Request $request)
    {
        $category = $request->get('category');

        $checklists = QualityInspectionChecklist::getForCategory($category);

        return response()->json($checklists);
    }
}