<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\ItemCategory;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class ItemCategoryController extends Controller
{
    public function __construct()
    {
        $this->middleware('warehouse.access');
        $this->middleware('warehouse.permission:warehouse.item-categories.view')->only(['index', 'show']);
        $this->middleware('warehouse.permission:warehouse.item-categories.create')->only(['create', 'store']);
        $this->middleware('warehouse.permission:warehouse.item-categories.edit')->only(['edit', 'update']);
        $this->middleware('warehouse.permission:warehouse.item-categories.delete')->only(['destroy']);
    }

    /**
     * Display a listing of item categories.
     */
    public function index(Request $request)
    {
        if ($request->expectsJson()) {
            return $this->getDataForTable($request);
        }

        return view('warehouse.item-categories.index');
    }

    /**
     * Get data for DataTable.
     */
    private function getDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        $query = ItemCategory::query();

        // Apply search
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('category_name', 'LIKE', "%{$search}%")
                  ->orWhere('type_of_goods', 'LIKE', "%{$search}%")
                  ->orWhere('description', 'LIKE', "%{$search}%");
            });
        }

        // Apply status filter
        if ($request->filled('status')) {
            $query->where('status', $request->get('status'));
        }

        // Get total count
        $totalRecords = ItemCategory::count();
        $filteredRecords = $query->count();

        // Apply pagination
        $categories = $query->skip($start)
                           ->take($length)
                           ->orderBy('created_at', 'desc')
                           ->get();

        // Format data for DataTable
        $data = $categories->map(function ($category) {
            return [
                'id' => $category->id,
                'category_name' => $category->category_name,
                'type_of_goods' => $category->type_of_goods,
                'description' => $category->description,
                'status' => $category->status,
                'items_count' => $category->active_items_count,
                'created_at' => $category->created_at->format('M d, Y H:i'),
                'created_by' => 'Admin', // You can add created_by field later
            ];
        });

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data
        ]);
    }

    /**
     * Show the form for creating a new item category.
     */
    public function create()
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.create
        return view('warehouse.item-categories.create');
    }

    /**
     * Store a newly created item category.
     */
    public function store(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.create

        $validatedData = $request->validate([
            'category_name' => 'required|string|max:255|unique:item_categories,category_name',
            'type_of_goods' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'status' => 'required|in:active,inactive',
        ]);

        try {
            $category = ItemCategory::create($validatedData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $category,
                    'message' => 'Item category created successfully'
                ], 201);
            }

            return redirect()->route('warehouse.item-categories.index')
                           ->with('success', 'Item category created successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Display the specified item category.
     */
    public function show($id)
    {
        $category = ItemCategory::with('items')->findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.view

        return view('warehouse.item-categories.show', compact('category'));
    }

    /**
     * Show the form for editing the specified item category.
     */
    public function edit($id)
    {
        $category = ItemCategory::findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.edit

        return view('warehouse.item-categories.edit', compact('category'));
    }

    /**
     * Update the specified item category.
     */
    public function update(Request $request, $id)
    {
        $category = ItemCategory::findOrFail($id);

        if (!$category) {
            abort(404, 'Item category not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.edit

        $validatedData = $request->validate([
            'category_name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('item_categories', 'category_name')->ignore($id)
            ],
            'type_of_goods' => 'required|string|max:255',
            'description' => 'nullable|string|max:1000',
            'status' => 'required|in:active,inactive',
        ]);

        try {
            $category->update($validatedData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $category,
                    'message' => 'Item category updated successfully'
                ]);
            }

            return redirect()->route('warehouse.item-categories.show', $category)
                           ->with('success', 'Item category updated successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified item category.
     */
    public function destroy($id)
    {
        $category = ItemCategory::findOrFail($id);
        // Authorization is handled by middleware: warehouse.permission:warehouse.item-categories.delete

        try {
            // Check if category has items
            if ($category->items()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete category that has items associated with it.'
                ], 400);
            }

            $category->delete();

            return response()->json([
                'success' => true,
                'message' => 'Item category deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
}