<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Services\Warehouse\ItemService;
use App\Services\Auth\RoleBasedAccessService;
use Illuminate\Http\Request;

class ItemController extends Controller
{
    protected $itemService;
    protected $accessService;

    public function __construct(ItemService $itemService, RoleBasedAccessService $accessService)
    {
        $this->itemService = $itemService;
        $this->accessService = $accessService;

        $this->middleware('warehouse.access');
        $this->middleware('warehouse.permission:warehouse.items.view')->only(['index', 'show']);
        $this->middleware('warehouse.permission:warehouse.items.create')->only(['create', 'store']);
        $this->middleware('warehouse.permission:warehouse.items.edit')->only(['edit', 'update']);
        $this->middleware('warehouse.permission:warehouse.items.delete')->only(['destroy']);
    }

    /**
     * Display a listing of the items.
     */
    public function index(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.view

        if ($request->expectsJson()) {
            return $this->getDataForTable($request);
        }

        return view('warehouse.items.index');
    }

    /**
     * Get data for DataTable.
     */
    private function getDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        $query = \App\Models\Warehouse\Item::with('category');

        // Apply search
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('item_code', 'LIKE', "%{$search}%")
                  ->orWhere('item_description', 'LIKE', "%{$search}%")
                  ->orWhereHas('category', function ($q) use ($search) {
                      $q->where('category_name', 'LIKE', "%{$search}%");
                  });
            });
        }

        // Apply filters
        if ($request->filled('status')) {
            $query->where('status', $request->get('status'));
        }
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->get('category_id'));
        }

        // Get total count
        $totalRecords = \App\Models\Warehouse\Item::count();
        $filteredRecords = $query->count();

        // Apply pagination
        $items = $query->skip($start)
                      ->take($length)
                      ->orderBy('created_at', 'desc')
                      ->get();

        // Format data for DataTable
        $data = $items->map(function ($item) {
            return [
                'id' => $item->id,
                'item_code' => $item->item_code,
                'item_description' => $item->item_description,
                'category_name' => $item->category ? $item->category->category_name : 'No Category',
                'unit_of_measure' => $item->unit_of_measure,
                'status' => $item->status,
                'min_stock_level' => $item->min_stock_level,
                'max_stock_level' => $item->max_stock_level,
                'reorder_point' => $item->reorder_point,
                'created_at' => $item->created_at->format('M d, Y H:i'),
                'created_by' => 'Admin', // You can add created_by field later
            ];
        });

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data
        ]);
    }

    /**
     * Show the form for creating a new item.
     */
    public function create()
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        $categories = \App\Models\Warehouse\ItemCategory::where('status', 'active')->get();

        return view('warehouse.items.create', compact('categories'));
    }

    /**
     * Store a newly created item in storage.
     */
    public function store(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        $validatedData = $request->validate([
            'item_code' => 'nullable|string|max:50|unique:items,item_code',
            'item_description' => 'required|string|max:255',
            'category_id' => 'required|exists:item_categories,id',
            'unit_of_measure' => 'required|string|max:50',
            'min_stock_level' => 'nullable|numeric|min:0',
            'max_stock_level' => 'nullable|numeric|min:0',
            'reorder_point' => 'nullable|numeric|min:0',
            'status' => 'required|in:active,inactive,discontinued',
            'notes' => 'nullable|string|max:1000'
        ]);

        // Generate item code if not provided
        if (empty($validatedData['item_code'])) {
            $validatedData['item_code'] = $this->generateItemCode($validatedData['category_id']);
        }

        try {
            $item = \App\Models\Warehouse\Item::create($validatedData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $item->load('category'),
                    'message' => 'Item created successfully'
                ], 201);
            }

            return redirect()->route('warehouse.items.index')
                           ->with('success', 'Item created successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Generate item code based on category
     */
    private function generateItemCode($categoryId)
    {
        $category = \App\Models\Warehouse\ItemCategory::find($categoryId);
        $prefix = $category ? strtoupper(substr($category->category_name, 0, 3)) : 'ITM';
        $timestamp = now()->format('ymd');
        $sequence = str_pad(\App\Models\Warehouse\Item::where('item_code', 'LIKE', $prefix . '%')->count() + 1, 3, '0', STR_PAD_LEFT);

        return $prefix . $timestamp . $sequence;
    }

    /**
     * Display the specified item.
     */
    public function show($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.view

        $item->load(['category', 'inventories', 'stockMovements.user']);

        return view('warehouse.items.show', compact('item'));
    }

    /**
     * Show the form for editing the specified item.
     */
    public function edit($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.edit

        $categories = \App\Models\Warehouse\ItemCategory::where('status', 'active')->get();

        return view('warehouse.items.edit', compact('item', 'categories'));
    }

    /**
     * Update the specified item in storage.
     */
    public function update(Request $request, $id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.edit

        $request->validate([
            'name' => 'required|string|max:255',
            'category_id' => 'nullable|exists:item_categories,id',
            'description' => 'nullable|string',
            'unit' => 'required|string|max:50',
            'unit_cost' => 'nullable|numeric|min:0',
            'selling_price' => 'nullable|numeric|min:0',
            'reorder_point' => 'nullable|integer|min:0',
            'max_stock_level' => 'nullable|integer|min:0',
            'location' => 'nullable|string|max:255',
            'status' => 'required|in:active,inactive'
        ]);

        try {
            $updatedItem = $this->itemService->update($id, $request->validated());

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $updatedItem,
                    'message' => 'Item updated successfully'
                ]);
            }

            return redirect()->route('warehouse.items.show', $updatedItem)
                           ->with('success', 'Item updated successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified item from storage.
     */
    public function destroy($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.delete

        try {
            $this->itemService->delete($id);

            return response()->json([
                'success' => true,
                'message' => 'Item deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Generate barcode for item.
     */
    public function generateBarcode($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        try {
            $barcodeUrl = $this->itemService->generateBarcode($item);

            return response()->json([
                'success' => true,
                'barcode_url' => $barcodeUrl,
                'message' => 'Barcode generated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Generate QR code for item.
     */
    public function generateQrCode($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        try {
            $qrCodeUrl = $this->itemService->generateQrCode($item);

            return response()->json([
                'success' => true,
                'qr_code_url' => $qrCodeUrl,
                'message' => 'QR code generated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Bulk import items.
     */
    public function bulkImport(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        $request->validate([
            'items' => 'required|array|min:1',
            'items.*.name' => 'required|string|max:255',
            'items.*.category_id' => 'nullable|exists:item_categories,id',
            'items.*.unit' => 'required|string|max:50',
        ]);

        try {
            $results = $this->itemService->bulkImport($request->input('items'));

            return response()->json([
                'success' => true,
                'data' => $results,
                'message' => "Successfully imported {$results['success']} items"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Export items.
     */
    public function export(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.view

        $filters = $request->only([
            'category_id', 'status', 'location', 'low_stock', 'out_of_stock',
            'created_from', 'created_to'
        ]);

        try {
            $itemsData = $this->itemService->exportItems($filters);

            return response()->json([
                'success' => true,
                'data' => $itemsData,
                'message' => 'Items exported successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }
}