<?php

namespace App\Models\Warehouse;

use App\Models\Department;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class IncomingOperation extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'operation_number',
        'operation_type',
        'operation_date',
        'status',

        // Reference Information
        'supplier_id',
        'project_id',
        'project_phase',
        'division_id',
        'site_location',
        'gps_coordinates',

        // LPO Management
        'lpo_number',
        'lpo_status',
        'lpo_budget_amount',
        'currency',
        'exchange_rate',

        // Delivery Information
        'delivery_note_number',
        'vehicle_number',
        'driver_name',
        'driver_contact',
        'delivery_datetime',
        'delivery_instructions',

        // Quality Control
        'requires_inspection',
        'inspector_id',
        'inspection_datetime',
        'inspection_result',
        'inspection_notes',
        'inspection_checklist',

        // Financial Information
        'total_amount',
        'total_amount_aed',
        'vat_amount',
        'vat_rate',

        // Documents
        'attached_documents',
        'delivery_note_path',
        'photo_documentation',

        // Workflow
        'received_by',
        'received_by_name',
        'approved_by',
        'approved_at',
        'approval_notes',

        // Compliance & Audit
        'regulatory_compliant',
        'compliance_checklist',
        'notes',
        'rejection_reason'
    ];

    protected $casts = [
        'operation_date' => 'date',
        'delivery_datetime' => 'datetime',
        'inspection_datetime' => 'datetime',
        'approved_at' => 'datetime',
        'total_amount' => 'decimal:2',
        'total_amount_aed' => 'decimal:2',
        'vat_amount' => 'decimal:2',
        'vat_rate' => 'decimal:2',
        'lpo_budget_amount' => 'decimal:2',
        'exchange_rate' => 'decimal:4',
        'requires_inspection' => 'boolean',
        'regulatory_compliant' => 'boolean',
        'gps_coordinates' => 'array',
        'inspection_checklist' => 'array',
        'attached_documents' => 'array',
        'photo_documentation' => 'array',
        'compliance_checklist' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($operation) {
            if (empty($operation->operation_number)) {
                $operation->operation_number = static::generateOperationNumber();
            }
        });

        static::saving(function ($operation) {
            // Auto-calculate VAT and AED amounts
            $operation->vat_amount = $operation->total_amount * ($operation->vat_rate / 100);

            if ($operation->currency !== 'AED') {
                $operation->total_amount_aed = $operation->total_amount * $operation->exchange_rate;
            } else {
                $operation->total_amount_aed = $operation->total_amount;
            }
        });
    }

    /**
     * Generate operation number
     */
    public static function generateOperationNumber()
    {
        $date = now()->format('Ymd');
        $lastOperation = static::whereDate('created_at', now())->latest()->first();
        $sequence = $lastOperation ? (int) substr($lastOperation->operation_number, -4) + 1 : 1;

        return 'INC-' . $date . '-' . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Get the supplier
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * Get the project
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    /**
     * Get the division
     */
    public function division(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'division_id');
    }

    /**
     * Get the receiver
     */
    public function receiver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'received_by');
    }

    /**
     * Get the inspector
     */
    public function inspector(): BelongsTo
    {
        return $this->belongsTo(User::class, 'inspector_id');
    }

    /**
     * Get the approver
     */
    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Get all items in this operation
     */
    public function items(): HasMany
    {
        return $this->hasMany(IncomingOperationItem::class);
    }

    /**
     * Get related LPO
     */
    public function lpo(): BelongsTo
    {
        return $this->belongsTo(LpoManagement::class, 'lpo_number', 'lpo_number');
    }

    /**
     * Scope for operation type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('operation_type', $type);
    }

    /**
     * Scope for status
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for date range
     */
    public function scopeDateBetween($query, $startDate, $endDate)
    {
        return $query->whereBetween('operation_date', [$startDate, $endDate]);
    }

    /**
     * Scope for project
     */
    public function scopeForProject($query, $projectId)
    {
        return $query->where('project_id', $projectId);
    }

    /**
     * Check if operation requires inspection
     */
    public function needsInspection()
    {
        return $this->requires_inspection && $this->inspection_result === null;
    }

    /**
     * Check if operation is approved
     */
    public function isApproved()
    {
        return $this->status === 'approved';
    }

    /**
     * Check if operation is completed
     */
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    /**
     * Get status badge class
     */
    public function getStatusBadgeClassAttribute()
    {
        return match($this->status) {
            'draft' => 'badge-secondary',
            'pending_inspection' => 'badge-warning',
            'approved' => 'badge-success',
            'rejected' => 'badge-danger',
            'completed' => 'badge-primary',
            default => 'badge-secondary'
        };
    }

    /**
     * Get operation type label
     */
    public function getOperationTypeLabelAttribute()
    {
        return match($this->operation_type) {
            'supplier_delivery' => 'Supplier Delivery',
            'site_return' => 'Site Return',
            'inter_division_transfer' => 'Inter-Division Transfer',
            'emergency_procurement' => 'Emergency Procurement',
            default => ucfirst(str_replace('_', ' ', $this->operation_type))
        };
    }

    /**
     * Get formatted total amount
     */
    public function getFormattedTotalAmountAttribute()
    {
        return $this->currency . ' ' . number_format($this->total_amount, 2);
    }

    /**
     * Get formatted total amount in AED
     */
    public function getFormattedTotalAmountAedAttribute()
    {
        return 'AED ' . number_format($this->total_amount_aed, 2);
    }

    /**
     * Complete the operation
     */
    public function complete()
    {
        if ($this->status !== 'approved') {
            throw new \Exception('Operation must be approved before completion');
        }

        $this->status = 'completed';
        $this->save();

        // Update inventory for each item
        foreach ($this->items as $item) {
            $this->updateInventory($item);
            $this->createStockMovement($item);
        }

        // Update LPO if exists
        if ($this->lpo) {
            $this->lpo->updateDeliveryStatus();
        }
    }

    /**
     * Update inventory for an item
     */
    protected function updateInventory($item)
    {
        $inventory = Inventory::firstOrNew([
            'item_id' => $item->item_id,
            'division_id' => $this->division_id,
            'batch_number' => $item->batch_number,
        ]);

        $inventory->quantity_available += $item->quantity_accepted;
        $inventory->unit_price = $item->unit_price_aed;
        $inventory->manufacturing_date = $item->manufacturing_date;
        $inventory->expiry_date = $item->expiry_date;
        $inventory->save();
    }

    /**
     * Create stock movement record
     */
    protected function createStockMovement($item)
    {
        StockMovement::create([
            'item_id' => $item->item_id,
            'movement_type' => 'in',
            'reference_type' => $this->operation_type,
            'reference_id' => $this->id,
            'quantity_before' => 0, // Should be calculated
            'quantity_moved' => $item->quantity_accepted,
            'quantity_after' => 0, // Should be calculated
            'division_id' => $this->division_id,
            'user_id' => $this->received_by,
            'notes' => "Incoming operation: {$this->operation_number}"
        ]);
    }
}