<?php

namespace App\Models\Warehouse;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ItemCategory extends Model
{
    use HasFactory;

    protected $table = 'item_categories';

    protected $fillable = [
        'category_name',
        'type_of_goods',
        'description',
        'status'
    ];

    protected $casts = [
        'status' => 'string',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Get all items in this category
     */
    public function items(): HasMany
    {
        return $this->hasMany(Item::class, 'category_id');
    }

    /**
     * Scope to get only active categories
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    /**
     * Get the count of active items in this category
     */
    public function getActiveItemsCountAttribute()
    {
        return $this->items()->where('status', 'active')->count();
    }

    /**
     * Get total stock value for this category
     */
    public function getTotalStockValueAttribute()
    {
        return $this->items()
            ->with('inventories')
            ->get()
            ->sum(function ($item) {
                return $item->inventories->sum('total_value');
            });
    }

    /**
     * Check if category can be deleted (no items associated)
     */
    public function getCanBeDeletedAttribute()
    {
        return $this->items()->count() === 0;
    }
}