<?php

namespace App\Models\Warehouse;

use App\Models\Department;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class OutgoingTransaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'transaction_date',
        'project_id',
        'division_id',
        'material_request_number',
        'type_of_goods',
        'requested_by',
        'approved_by',
        'receiver_name',
        'receiver_signature',
        'status',
        'notes'
    ];

    protected $casts = [
        'transaction_date' => 'date',
        'status' => 'string',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Get the project for the transaction
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    /**
     * Get the division for the transaction
     */
    public function division(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'division_id');
    }

    /**
     * Get the user who requested the materials
     */
    public function requester(): BelongsTo
    {
        return $this->belongsTo(User::class, 'requested_by');
    }

    /**
     * Get the user who approved the request
     */
    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Get all items in this transaction
     */
    public function items(): HasMany
    {
        return $this->hasMany(OutgoingItem::class);
    }

    /**
     * Get the material delivery note for this transaction
     */
    public function deliveryNote(): HasOne
    {
        return $this->hasOne(MaterialDeliveryNote::class);
    }

    /**
     * Get stock movements created by this transaction
     */
    public function stockMovements()
    {
        return $this->morphMany(StockMovement::class, 'reference');
    }

    /**
     * Scope for pending transactions
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for approved transactions
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * Scope for completed transactions
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope for transactions by project
     */
    public function scopeForProject($query, $projectId)
    {
        return $query->where('project_id', $projectId);
    }

    /**
     * Approve the transaction
     */
    public function approve($userId)
    {
        $this->approved_by = $userId;
        $this->status = 'approved';
        $this->save();
    }

    /**
     * Complete the transaction and update inventory
     */
    public function complete()
    {
        $this->status = 'completed';
        $this->save();

        // Update inventory for each item
        foreach ($this->items as $item) {
            $this->updateInventory($item);
            $this->createStockMovement($item);
        }
    }

    /**
     * Update inventory for an item
     */
    protected function updateInventory($outgoingItem)
    {
        $inventories = Inventory::where('item_id', $outgoingItem->item_id)
            ->where('division_id', $this->division_id)
            ->where('quantity_available', '>', 0)
            ->orderBy('expiry_date', 'asc') // FIFO by expiry date
            ->get();

        $remainingQuantity = $outgoingItem->quantity_released;

        foreach ($inventories as $inventory) {
            if ($remainingQuantity <= 0) break;

            $quantityToDeduct = min($inventory->quantity_available, $remainingQuantity);
            $inventory->updateQuantity($quantityToDeduct, 'subtract');
            $remainingQuantity -= $quantityToDeduct;
        }
    }

    /**
     * Create stock movement record
     */
    protected function createStockMovement($outgoingItem)
    {
        StockMovement::create([
            'item_id' => $outgoingItem->item_id,
            'movement_type' => 'out',
            'reference_type' => 'outgoing',
            'reference_id' => $this->id,
            'quantity_before' => 0, // This should be calculated
            'quantity_moved' => $outgoingItem->quantity_released,
            'quantity_after' => 0, // This should be calculated
            'division_id' => $this->division_id,
            'user_id' => $this->requested_by,
            'notes' => "Outgoing to project: {$this->project->project_name}"
        ]);
    }

    /**
     * Check if transaction can be fulfilled
     */
    public function getCanBeFulfilledAttribute()
    {
        foreach ($this->items as $item) {
            $availableQuantity = Inventory::where('item_id', $item->item_id)
                ->where('division_id', $this->division_id)
                ->sum('quantity_available');

            if ($availableQuantity < $item->quantity_requested) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get transaction reference number
     */
    public function getReferenceNumberAttribute()
    {
        return $this->material_request_number ?: 'OUT-' . str_pad($this->id, 6, '0', STR_PAD_LEFT);
    }
}