<?php

namespace App\Policies\Warehouse;

use App\Models\User;
use App\Services\Auth\RoleBasedAccessService;

abstract class BaseWarehousePolicy
{
    /**
     * The role-based access service.
     */
    protected RoleBasedAccessService $accessService;

    /**
     * Create a new policy instance.
     */
    public function __construct(RoleBasedAccessService $accessService)
    {
        $this->accessService = $accessService;
    }

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user): bool
    {
        return $this->accessService->hasWarehouseAccess($user);
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, $model): bool
    {
        return $this->accessService->hasWarehouseAccess($user);
    }

    /**
     * Check if user has specific warehouse permission.
     */
    protected function hasWarehousePermission(User $user, string $permission): bool
    {
        return $this->accessService->hasPermission($user, $permission);
    }

    /**
     * Check if user has any of the warehouse permissions.
     */
    protected function hasAnyWarehousePermission(User $user, array $permissions): bool
    {
        return $this->accessService->hasAnyPermission($user, $permissions);
    }

    /**
     * Check if user is warehouse manager or higher.
     */
    protected function isWarehouseManagerOrHigher(User $user): bool
    {
        $roleLevel = $this->accessService->getWarehouseRoleLevel($user);
        return in_array($roleLevel, ['manager']) || $this->accessService->isAdmin($user);
    }

    /**
     * Check if user is warehouse supervisor or higher.
     */
    protected function isWarehouseSupervisorOrHigher(User $user): bool
    {
        $roleLevel = $this->accessService->getWarehouseRoleLevel($user);
        return in_array($roleLevel, ['manager', 'supervisor']) || $this->accessService->isAdmin($user);
    }

    /**
     * Check if user can perform operation on model.
     */
    protected function canPerformOperation(User $user, string $operation, string $module): bool
    {
        return $this->accessService->canPerformWarehouseOperation($user, $operation, $module);
    }
}