<?php

namespace App\Policies\Warehouse;

use App\Models\User;
use App\Models\Warehouse\IncomingTransaction;
use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\ReturnTransaction;

class TransactionPolicy extends BaseWarehousePolicy
{
    /**
     * Determine whether the user can view any incoming transactions.
     */
    public function viewAnyIncoming(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.incoming.view');
    }

    /**
     * Determine whether the user can view the incoming transaction.
     */
    public function viewIncoming(User $user, IncomingTransaction $transaction): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.incoming.view');
    }

    /**
     * Determine whether the user can create incoming transactions.
     */
    public function createIncoming(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.incoming.create');
    }

    /**
     * Determine whether the user can update the incoming transaction.
     */
    public function updateIncoming(User $user, IncomingTransaction $transaction): bool
    {
        // Can't edit completed transactions
        if ($transaction->status === 'completed') {
            return false;
        }

        return $this->hasWarehousePermission($user, 'warehouse.incoming.edit');
    }

    /**
     * Determine whether the user can delete the incoming transaction.
     */
    public function deleteIncoming(User $user, IncomingTransaction $transaction): bool
    {
        // Can't delete completed transactions
        if ($transaction->status === 'completed') {
            return false;
        }

        return $this->hasWarehousePermission($user, 'warehouse.incoming.delete') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Determine whether the user can process incoming transactions.
     */
    public function processIncoming(User $user, IncomingTransaction $transaction): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.incoming.process');
    }

    /**
     * Determine whether the user can view any outgoing transactions.
     */
    public function viewAnyOutgoing(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.outgoing.view');
    }

    /**
     * Determine whether the user can view the outgoing transaction.
     */
    public function viewOutgoing(User $user, OutgoingTransaction $transaction): bool
    {
        // Project managers can only view their own project transactions
        $roleLevel = $this->accessService->getWarehouseRoleLevel($user);
        if ($roleLevel === 'project_manager') {
            return $transaction->project_id &&
                   $this->canAccessProject($user, $transaction->project_id);
        }

        return $this->hasWarehousePermission($user, 'warehouse.outgoing.view');
    }

    /**
     * Determine whether the user can create outgoing transactions.
     */
    public function createOutgoing(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.outgoing.create');
    }

    /**
     * Determine whether the user can update the outgoing transaction.
     */
    public function updateOutgoing(User $user, OutgoingTransaction $transaction): bool
    {
        // Can't edit dispatched transactions
        if ($transaction->status === 'dispatched') {
            return false;
        }

        // Project managers can only edit their own project transactions
        $roleLevel = $this->accessService->getWarehouseRoleLevel($user);
        if ($roleLevel === 'project_manager') {
            return $transaction->project_id &&
                   $this->canAccessProject($user, $transaction->project_id) &&
                   $this->hasWarehousePermission($user, 'warehouse.outgoing.edit');
        }

        return $this->hasWarehousePermission($user, 'warehouse.outgoing.edit');
    }

    /**
     * Determine whether the user can delete the outgoing transaction.
     */
    public function deleteOutgoing(User $user, OutgoingTransaction $transaction): bool
    {
        // Can't delete dispatched transactions
        if ($transaction->status === 'dispatched') {
            return false;
        }

        return $this->hasWarehousePermission($user, 'warehouse.outgoing.delete') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Determine whether the user can approve outgoing transactions.
     */
    public function approveOutgoing(User $user, OutgoingTransaction $transaction): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.outgoing.approve') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Determine whether the user can dispatch outgoing transactions.
     */
    public function dispatchOutgoing(User $user, OutgoingTransaction $transaction): bool
    {
        // Must be approved first
        if ($transaction->status !== 'approved') {
            return false;
        }

        return $this->hasWarehousePermission($user, 'warehouse.outgoing.process');
    }

    /**
     * Determine whether the user can view any return transactions.
     */
    public function viewAnyReturns(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.returns.view');
    }

    /**
     * Determine whether the user can view the return transaction.
     */
    public function viewReturn(User $user, ReturnTransaction $transaction): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.returns.view');
    }

    /**
     * Determine whether the user can create return transactions.
     */
    public function createReturn(User $user): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.returns.create');
    }

    /**
     * Determine whether the user can update the return transaction.
     */
    public function updateReturn(User $user, ReturnTransaction $transaction): bool
    {
        // Can't edit processed returns
        if ($transaction->status === 'processed') {
            return false;
        }

        return $this->hasWarehousePermission($user, 'warehouse.returns.edit');
    }

    /**
     * Determine whether the user can delete the return transaction.
     */
    public function deleteReturn(User $user, ReturnTransaction $transaction): bool
    {
        // Can't delete processed returns
        if ($transaction->status === 'processed') {
            return false;
        }

        return $this->hasWarehousePermission($user, 'warehouse.returns.delete') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Determine whether the user can process return transactions.
     */
    public function processReturn(User $user, ReturnTransaction $transaction): bool
    {
        return $this->hasWarehousePermission($user, 'warehouse.returns.process') &&
               $this->isWarehouseSupervisorOrHigher($user);
    }

    /**
     * Check if user can access specific project.
     */
    private function canAccessProject(User $user, int $projectId): bool
    {
        // This would check if the user is assigned to the project
        // Implementation depends on your project assignment logic
        return true; // Simplified for now
    }
}