<?php

namespace App\Repositories\Warehouse;

use App\Repositories\BaseRepository;
use App\Models\Warehouse\ItemCategory;
use Illuminate\Database\Eloquent\Builder;

class ItemCategoryRepository extends BaseRepository
{
    /**
     * Create a new repository instance.
     */
    public function __construct(ItemCategory $model)
    {
        $this->model = $model;
    }

    /**
     * Get categories with item counts.
     */
    public function getWithItemCounts()
    {
        return $this->model->withCount('items')->get();
    }

    /**
     * Get active categories.
     */
    public function getActive()
    {
        return $this->model->where('status', 'active')
                          ->orderBy('name')
                          ->get();
    }

    /**
     * Find category by code.
     */
    public function findByCode(string $code): ?ItemCategory
    {
        return $this->model->where('code', $code)->first();
    }

    /**
     * Get categories with hierarchy.
     */
    public function getHierarchy()
    {
        return $this->model->whereNull('parent_id')
                          ->with('children')
                          ->orderBy('name')
                          ->get();
    }

    /**
     * Apply category-specific search filters.
     */
    protected function applySearchFilter(Builder $query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
              ->orWhere('code', 'LIKE', "%{$search}%")
              ->orWhere('description', 'LIKE', "%{$search}%");
        });
    }

    /**
     * Apply category-specific filters.
     */
    protected function applyFilters(Builder $query, array $filters)
    {
        foreach ($filters as $key => $value) {
            if ($value === null || $value === '') {
                continue;
            }

            switch ($key) {
                case 'status':
                    $query->where('status', $value);
                    break;
                case 'parent_id':
                    if ($value === 'null') {
                        $query->whereNull('parent_id');
                    } else {
                        $query->where('parent_id', $value);
                    }
                    break;
                case 'search':
                    $this->applySearchFilter($query, $value);
                    break;
            }
        }

        return $query;
    }

    /**
     * Get category statistics.
     */
    public function getStatistics()
    {
        return [
            'total_categories' => $this->model->count(),
            'active_categories' => $this->model->where('status', 'active')->count(),
            'parent_categories' => $this->model->whereNull('parent_id')->count(),
            'subcategories' => $this->model->whereNotNull('parent_id')->count(),
        ];
    }
}