<?php

namespace App\Services\Warehouse;

use App\Services\BaseService;
use App\Repositories\Warehouse\InventoryRepository;
use App\Models\Warehouse\Inventory;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\StockMovement;
use Illuminate\Support\Facades\DB;

class InventoryService extends BaseService
{
    /**
     * Create a new service instance.
     */
    public function __construct(InventoryRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Adjust inventory levels.
     */
    public function adjustInventory(int $itemId, int $quantity, string $type, string $reason, array $metadata = [])
    {
        DB::beginTransaction();
        try {
            $item = Item::findOrFail($itemId);

            // Get or create inventory record
            $inventory = $this->repository->findByItemId($itemId) ??
                        $this->repository->create([
                            'item_id' => $itemId,
                            'quantity_available' => 0,
                            'quantity_reserved' => 0,
                            'unit_cost' => $item->unit_cost ?? 0,
                        ]);

            $oldQuantity = $inventory->quantity_available;

            // Apply adjustment
            if ($type === 'increase') {
                $inventory->quantity_available += $quantity;
            } elseif ($type === 'decrease') {
                if ($inventory->quantity_available < $quantity) {
                    throw new \Exception("Insufficient stock. Available: {$inventory->quantity_available}, Requested: {$quantity}");
                }
                $inventory->quantity_available -= $quantity;
            } elseif ($type === 'set') {
                if ($quantity < 0) {
                    throw new \Exception("Quantity cannot be negative.");
                }
                $inventory->quantity_available = $quantity;
            } else {
                throw new \Exception("Invalid adjustment type: {$type}");
            }

            $inventory->save();

            // Update item totals
            $this->updateItemTotals($item);

            // Record stock movement
            $this->recordStockMovement($item, $type, $oldQuantity, $inventory->quantity_available, $reason, $metadata);

            DB::commit();
            return $inventory;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Transfer inventory between locations.
     */
    public function transferInventory(int $itemId, string $fromLocation, string $toLocation, int $quantity, string $reason = '')
    {
        DB::beginTransaction();
        try {
            // Get source inventory
            $sourceInventory = $this->repository->findByItemAndLocation($itemId, $fromLocation);
            if (!$sourceInventory || $sourceInventory->quantity_available < $quantity) {
                throw new \Exception("Insufficient stock at source location.");
            }

            // Get or create destination inventory
            $destinationInventory = $this->repository->findByItemAndLocation($itemId, $toLocation) ??
                                  $this->repository->create([
                                      'item_id' => $itemId,
                                      'location' => $toLocation,
                                      'quantity_available' => 0,
                                      'quantity_reserved' => 0,
                                      'unit_cost' => $sourceInventory->unit_cost,
                                  ]);

            // Perform transfer
            $sourceInventory->quantity_available -= $quantity;
            $destinationInventory->quantity_available += $quantity;

            $sourceInventory->save();
            $destinationInventory->save();

            // Record stock movements
            $item = Item::find($itemId);
            $this->recordStockMovement($item, 'transfer_out', $sourceInventory->quantity_available + $quantity,
                                     $sourceInventory->quantity_available, $reason, [
                'from_location' => $fromLocation,
                'to_location' => $toLocation,
                'transfer_type' => 'outbound'
            ]);

            $this->recordStockMovement($item, 'transfer_in', $destinationInventory->quantity_available - $quantity,
                                     $destinationInventory->quantity_available, $reason, [
                'from_location' => $fromLocation,
                'to_location' => $toLocation,
                'transfer_type' => 'inbound'
            ]);

            DB::commit();
            return [
                'source' => $sourceInventory,
                'destination' => $destinationInventory
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Reserve inventory quantity.
     */
    public function reserveInventory(int $itemId, int $quantity, string $reason = '', array $metadata = [])
    {
        DB::beginTransaction();
        try {
            $inventory = $this->repository->findByItemId($itemId);
            if (!$inventory) {
                throw new \Exception("No inventory found for item.");
            }

            if ($inventory->quantity_available < $quantity) {
                throw new \Exception("Insufficient available quantity. Available: {$inventory->quantity_available}, Requested: {$quantity}");
            }

            // Move quantity from available to reserved
            $inventory->quantity_available -= $quantity;
            $inventory->quantity_reserved += $quantity;
            $inventory->save();

            // Update item totals
            $item = Item::find($itemId);
            $this->updateItemTotals($item);

            // Record stock movement
            $this->recordStockMovement($item, 'reserve', $inventory->quantity_available + $quantity,
                                     $inventory->quantity_available, $reason, $metadata);

            DB::commit();
            return $inventory;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Release reserved inventory.
     */
    public function releaseReservedInventory(int $itemId, int $quantity, string $reason = '', array $metadata = [])
    {
        DB::beginTransaction();
        try {
            $inventory = $this->repository->findByItemId($itemId);
            if (!$inventory) {
                throw new \Exception("No inventory found for item.");
            }

            if ($inventory->quantity_reserved < $quantity) {
                throw new \Exception("Insufficient reserved quantity. Reserved: {$inventory->quantity_reserved}, Requested: {$quantity}");
            }

            // Move quantity from reserved to available
            $inventory->quantity_reserved -= $quantity;
            $inventory->quantity_available += $quantity;
            $inventory->save();

            // Update item totals
            $item = Item::find($itemId);
            $this->updateItemTotals($item);

            // Record stock movement
            $this->recordStockMovement($item, 'release', $inventory->quantity_available - $quantity,
                                     $inventory->quantity_available, $reason, $metadata);

            DB::commit();
            return $inventory;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Perform physical count and adjust discrepancies.
     */
    public function performPhysicalCount(array $countData)
    {
        $results = [
            'total_items' => 0,
            'adjustments' => 0,
            'discrepancies' => []
        ];

        DB::beginTransaction();
        try {
            foreach ($countData as $data) {
                $itemId = $data['item_id'];
                $countedQuantity = $data['counted_quantity'];
                $location = $data['location'] ?? null;

                $inventory = $this->repository->findByItemId($itemId);
                if (!$inventory) {
                    continue;
                }

                $systemQuantity = $inventory->quantity_available;
                $discrepancy = $countedQuantity - $systemQuantity;

                $results['total_items']++;

                if ($discrepancy != 0) {
                    // Adjust inventory to match physical count
                    $this->adjustInventory($itemId, abs($discrepancy),
                                         $discrepancy > 0 ? 'increase' : 'decrease',
                                         'Physical count adjustment', [
                        'physical_count' => true,
                        'counted_quantity' => $countedQuantity,
                        'system_quantity' => $systemQuantity,
                        'discrepancy' => $discrepancy,
                        'location' => $location
                    ]);

                    $results['adjustments']++;
                    $results['discrepancies'][] = [
                        'item_id' => $itemId,
                        'system_quantity' => $systemQuantity,
                        'counted_quantity' => $countedQuantity,
                        'discrepancy' => $discrepancy
                    ];
                }
            }

            DB::commit();
            return $results;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Get low stock items.
     */
    public function getLowStockItems()
    {
        return $this->repository->getLowStockItems();
    }

    /**
     * Get inventory by location.
     */
    public function getInventoryByLocation(string $location)
    {
        return $this->repository->getByLocation($location);
    }

    /**
     * Get inventory aging report.
     */
    public function getInventoryAging(int $days = 90)
    {
        return $this->repository->getAgingInventory($days);
    }

    /**
     * Check availability for multiple items.
     */
    public function checkBulkAvailability(array $items): array
    {
        $results = [];

        foreach ($items as $itemData) {
            $itemId = $itemData['item_id'];
            $requiredQuantity = $itemData['quantity'];

            $inventory = $this->repository->findByItemId($itemId);
            $availableQuantity = $inventory ? $inventory->quantity_available : 0;

            $results[] = [
                'item_id' => $itemId,
                'required_quantity' => $requiredQuantity,
                'available_quantity' => $availableQuantity,
                'is_available' => $availableQuantity >= $requiredQuantity,
                'shortage' => max(0, $requiredQuantity - $availableQuantity)
            ];
        }

        return $results;
    }

    /**
     * Update item total quantities.
     */
    protected function updateItemTotals(Item $item)
    {
        $totalAvailable = $item->inventories()->sum('quantity_available');
        $totalReserved = $item->inventories()->sum('quantity_reserved');

        $item->update([
            'total_quantity' => $totalAvailable + $totalReserved,
            'available_quantity' => $totalAvailable,
            'reserved_quantity' => $totalReserved
        ]);
    }

    /**
     * Record stock movement.
     */
    protected function recordStockMovement(Item $item, string $type, int $previousQuantity, int $newQuantity, string $reason, array $metadata = [])
    {
        StockMovement::create([
            'item_id' => $item->id,
            'type' => $type,
            'quantity' => abs($newQuantity - $previousQuantity),
            'previous_quantity' => $previousQuantity,
            'new_quantity' => $newQuantity,
            'reason' => $reason,
            'user_id' => auth()->id(),
            'metadata' => $metadata,
            'movable_type' => get_class($item),
            'movable_id' => $item->id
        ]);
    }
}