<?php

namespace App\Services\Warehouse;

use App\Services\BaseService;
use App\Repositories\Warehouse\SupplierRepository;
use App\Models\Warehouse\Supplier;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Carbon\Carbon;

class SupplierService extends BaseService
{
    /**
     * Create a new service instance.
     */
    public function __construct(SupplierRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Create a new supplier with automatic code generation.
     */
    public function create(array $data)
    {
        // Generate supplier code if not provided
        if (!isset($data['supplier_code'])) {
            $data['supplier_code'] = Supplier::generateSupplierCode();
        }

        // Set created_by
        $data['created_by'] = auth()->id();

        // Validate business rules
        $this->validateSupplierCreation($data);

        return parent::create($data);
    }

    /**
     * Update supplier with validation.
     */
    public function update($id, array $data)
    {
        $supplier = $this->findById($id);
        if (!$supplier) {
            throw new \Exception("Supplier not found with ID: {$id}");
        }

        // Don't allow changing supplier code if supplier has transactions
        if (isset($data['supplier_code']) && $data['supplier_code'] !== $supplier->supplier_code) {
            if ($supplier->incomingTransactions()->exists()) {
                throw new \Exception("Cannot change supplier code for suppliers with existing transactions.");
            }
        }

        // Validate business rules
        $this->validateSupplierUpdate($data, $supplier);

        return parent::update($id, $data);
    }

    /**
     * Get suppliers for DataTables with server-side processing.
     */
    public function getForDataTable(array $request)
    {
        $filters = [
            'search' => $request['search']['value'] ?? null,
            'status' => $request['status'] ?? null,
            'country' => $request['country'] ?? null,
            'city' => $request['city'] ?? null,
            'rating_min' => $request['rating_min'] ?? null,
            'rating_max' => $request['rating_max'] ?? null,
            'created_from' => $request['created_from'] ?? null,
            'created_to' => $request['created_to'] ?? null,
        ];

        $query = $this->repository->getForDataTable($filters);

        // Get total count before pagination
        $totalRecords = $query->count();

        // Apply ordering
        if (isset($request['order']) && count($request['order']) > 0) {
            $orderColumn = $request['columns'][$request['order'][0]['column']]['data'];
            $orderDirection = $request['order'][0]['dir'];

            // Map column names to database fields
            $columnMap = [
                'supplier_code' => 'supplier_code',
                'name' => 'name',
                'contact_person' => 'contact_person',
                'email' => 'email',
                'phone' => 'phone',
                'city' => 'city',
                'country' => 'country',
                'status' => 'status',
                'rating' => 'rating',
                'created_at' => 'created_at',
            ];

            if (isset($columnMap[$orderColumn])) {
                $query->orderBy($columnMap[$orderColumn], $orderDirection);
            }
        } else {
            $query->orderBy('created_at', 'desc');
        }

        // Apply pagination
        $start = $request['start'] ?? 0;
        $length = $request['length'] ?? 10;

        if ($length != -1) {
            $query->skip($start)->take($length);
        }

        $suppliers = $query->get();

        return [
            'draw' => $request['draw'] ?? 1,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $totalRecords,
            'data' => $suppliers->map(function ($supplier) {
                return [
                    'id' => $supplier->id,
                    'supplier_code' => $supplier->supplier_code,
                    'name' => $supplier->name,
                    'contact_person' => $supplier->contact_person,
                    'email' => $supplier->email,
                    'phone' => $supplier->phone,
                    'city' => $supplier->city,
                    'country' => $supplier->country,
                    'status' => $supplier->status,
                    'status_color' => $supplier->status_color,
                    'rating' => $supplier->formatted_rating,
                    'payment_terms' => $supplier->payment_terms,
                    'created_at' => $supplier->created_at->format('M d, Y'),
                    'created_by' => $supplier->createdBy?->name ?? 'System',
                    'performance_score' => number_format($supplier->performance_score, 1),
                ];
            })->toArray()
        ];
    }

    /**
     * Get supplier performance data.
     */
    public function getPerformanceData($supplierId)
    {
        $supplier = $this->findById($supplierId);
        if (!$supplier) {
            throw new \Exception("Supplier not found");
        }

        return [
            'supplier' => $supplier,
            'performance_score' => $supplier->performance_score,
            'total_orders' => $supplier->getTotalOrdersCount(),
            'total_value' => $supplier->getTotalOrderValue(),
            'average_order_value' => $supplier->getAverageOrderValue(),
            'recent_activity' => $supplier->getRecentActivity(),
            'top_items' => $supplier->getTopPurchasedItems(),
            'rating_breakdown' => $this->getRatingBreakdown($supplier),
            'monthly_performance' => $this->getMonthlyPerformance($supplier),
        ];
    }

    /**
     * Import suppliers from Excel file.
     */
    public function importFromExcel(UploadedFile $file): array
    {
        $results = [
            'success' => 0,
            'errors' => [],
            'warnings' => [],
            'created_suppliers' => []
        ];

        DB::beginTransaction();

        try {
            $spreadsheet = IOFactory::load($file->getPathname());
            $worksheet = $spreadsheet->getActiveSheet();
            $rows = $worksheet->toArray();

            // Remove header row
            $header = array_shift($rows);

            // Validate header
            $expectedHeaders = [
                'name', 'contact_person', 'email', 'phone', 'address',
                'city', 'state', 'country', 'postal_code', 'payment_terms'
            ];

            foreach ($rows as $index => $row) {
                $rowNumber = $index + 2; // +2 because we removed header and Excel rows start at 1

                try {
                    // Skip empty rows
                    if (empty(array_filter($row))) {
                        continue;
                    }

                    // Validate required fields
                    if (empty($row[0])) { // name
                        $results['errors'][] = "Row {$rowNumber}: Name is required";
                        continue;
                    }

                    // Check for duplicate supplier
                    $existingSupplier = Supplier::where('name', $row[0])
                                              ->orWhere('email', $row[2])
                                              ->first();

                    if ($existingSupplier) {
                        $results['warnings'][] = "Row {$rowNumber}: Supplier '{$row[0]}' already exists";
                        continue;
                    }

                    // Prepare data
                    $supplierData = [
                        'name' => $row[0],
                        'contact_person' => $row[1] ?? null,
                        'email' => $row[2] ?? null,
                        'phone' => $row[3] ?? null,
                        'address' => $row[4] ?? null,
                        'city' => $row[5] ?? null,
                        'state' => $row[6] ?? null,
                        'country' => $row[7] ?? null,
                        'postal_code' => $row[8] ?? null,
                        'payment_terms' => $row[9] ?? 'Net 30',
                        'status' => 'active',
                    ];

                    // Create supplier
                    $supplier = $this->create($supplierData);
                    $results['created_suppliers'][] = $supplier;
                    $results['success']++;

                } catch (\Exception $e) {
                    $results['errors'][] = "Row {$rowNumber}: " . $e->getMessage();
                }
            }

            DB::commit();

            return $results;

        } catch (\Exception $e) {
            DB::rollBack();
            throw new \Exception("Failed to import suppliers: " . $e->getMessage());
        }
    }

    /**
     * Export suppliers to Excel.
     */
    public function exportToExcel(array $filters = []): string
    {
        $suppliers = $this->getAll($filters);

        $spreadsheet = new Spreadsheet();
        $worksheet = $spreadsheet->getActiveSheet();

        // Set headers
        $headers = [
            'A1' => 'Supplier Code',
            'B1' => 'Name',
            'C1' => 'Contact Person',
            'D1' => 'Email',
            'E1' => 'Phone',
            'F1' => 'Address',
            'G1' => 'City',
            'H1' => 'State',
            'I1' => 'Country',
            'J1' => 'Postal Code',
            'K1' => 'Payment Terms',
            'L1' => 'Status',
            'M1' => 'Rating',
            'N1' => 'Created Date',
        ];

        foreach ($headers as $cell => $header) {
            $worksheet->setCellValue($cell, $header);
        }

        // Style headers
        $worksheet->getStyle('A1:N1')->getFont()->setBold(true);
        $worksheet->getStyle('A1:N1')->getFill()
                  ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                  ->getStartColor()->setRGB('E3F2FD');

        // Add data
        $row = 2;
        foreach ($suppliers as $supplier) {
            $worksheet->setCellValue("A{$row}", $supplier->supplier_code);
            $worksheet->setCellValue("B{$row}", $supplier->name);
            $worksheet->setCellValue("C{$row}", $supplier->contact_person);
            $worksheet->setCellValue("D{$row}", $supplier->email);
            $worksheet->setCellValue("E{$row}", $supplier->phone);
            $worksheet->setCellValue("F{$row}", $supplier->address);
            $worksheet->setCellValue("G{$row}", $supplier->city);
            $worksheet->setCellValue("H{$row}", $supplier->state);
            $worksheet->setCellValue("I{$row}", $supplier->country);
            $worksheet->setCellValue("J{$row}", $supplier->postal_code);
            $worksheet->setCellValue("K{$row}", $supplier->payment_terms);
            $worksheet->setCellValue("L{$row}", ucfirst($supplier->status));
            $worksheet->setCellValue("M{$row}", $supplier->formatted_rating);
            $worksheet->setCellValue("N{$row}", $supplier->created_at->format('Y-m-d H:i:s'));

            $row++;
        }

        // Auto-size columns
        foreach (range('A', 'N') as $column) {
            $worksheet->getColumnDimension($column)->setAutoSize(true);
        }

        // Save file
        $filename = 'suppliers_export_' . date('Y-m-d_H-i-s') . '.xlsx';
        $filePath = storage_path('app/temp/' . $filename);

        // Ensure temp directory exists
        if (!file_exists(dirname($filePath))) {
            mkdir(dirname($filePath), 0755, true);
        }

        $writer = new Xlsx($spreadsheet);
        $writer->save($filePath);

        return $filePath;
    }

    /**
     * Update supplier rating.
     */
    public function updateRating($supplierId, float $rating, string $notes = null)
    {
        if ($rating < 0 || $rating > 5) {
            throw new \Exception("Rating must be between 0 and 5");
        }

        $supplier = $this->findById($supplierId);
        if (!$supplier) {
            throw new \Exception("Supplier not found");
        }

        $supplier->update([
            'rating' => $rating,
            'notes' => $notes ?? $supplier->notes,
        ]);

        $this->logActivity('rating_updated', $supplier, [
            'old_rating' => $supplier->getOriginal('rating'),
            'new_rating' => $rating,
            'notes' => $notes,
        ]);

        return $supplier;
    }

    /**
     * Get supplier statistics for dashboard.
     */
    public function getDashboardStatistics()
    {
        return $this->repository->getStatistics();
    }

    /**
     * Get filter options for dropdowns.
     */
    public function getFilterOptions()
    {
        return [
            'countries' => $this->repository->getCountries(),
            'cities' => $this->repository->getCities(),
            'payment_terms' => $this->repository->getPaymentTerms(),
            'statuses' => ['active', 'inactive', 'suspended'],
        ];
    }

    /**
     * Get rating breakdown for supplier.
     */
    protected function getRatingBreakdown($supplier)
    {
        // This would typically come from review/rating system
        return [
            'overall' => $supplier->rating,
            'delivery' => 4.2,
            'quality' => 4.0,
            'communication' => 4.5,
            'pricing' => 3.8,
        ];
    }

    /**
     * Get monthly performance data.
     */
    protected function getMonthlyPerformance($supplier)
    {
        $months = [];
        for ($i = 5; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $months[] = [
                'month' => $date->format('M Y'),
                'orders' => rand(5, 20), // This would come from actual data
                'value' => rand(10000, 50000),
                'performance_score' => rand(70, 95),
            ];
        }

        return $months;
    }

    /**
     * Validate supplier creation.
     */
    protected function validateSupplierCreation(array $data)
    {
        // Check for duplicate supplier code
        if (isset($data['supplier_code'])) {
            $existingSupplier = Supplier::where('supplier_code', $data['supplier_code'])->first();
            if ($existingSupplier) {
                throw new \Exception("Supplier code '{$data['supplier_code']}' already exists.");
            }
        }

        // Check for duplicate email
        if (isset($data['email']) && !empty($data['email'])) {
            $existingSupplier = Supplier::where('email', $data['email'])->first();
            if ($existingSupplier) {
                throw new \Exception("Email '{$data['email']}' is already registered with another supplier.");
            }
        }

        // Validate rating
        if (isset($data['rating']) && ($data['rating'] < 0 || $data['rating'] > 5)) {
            throw new \Exception("Rating must be between 0 and 5.");
        }

        // Validate credit limit
        if (isset($data['credit_limit']) && $data['credit_limit'] < 0) {
            throw new \Exception("Credit limit cannot be negative.");
        }
    }

    /**
     * Validate supplier update.
     */
    protected function validateSupplierUpdate(array $data, Supplier $supplier)
    {
        // Check for duplicate supplier code (excluding current supplier)
        if (isset($data['supplier_code']) && $data['supplier_code'] !== $supplier->supplier_code) {
            $existingSupplier = Supplier::where('supplier_code', $data['supplier_code'])->first();
            if ($existingSupplier && $existingSupplier->id !== $supplier->id) {
                throw new \Exception("Supplier code '{$data['supplier_code']}' already exists.");
            }
        }

        // Check for duplicate email (excluding current supplier)
        if (isset($data['email']) && $data['email'] !== $supplier->email && !empty($data['email'])) {
            $existingSupplier = Supplier::where('email', $data['email'])->first();
            if ($existingSupplier && $existingSupplier->id !== $supplier->id) {
                throw new \Exception("Email '{$data['email']}' is already registered with another supplier.");
            }
        }

        // Other validation rules
        $this->validateSupplierCreation($data);
    }
}