/**
 * Admin Custom JavaScript
 * Custom functionality for the admin panel
 * Extracted from admin-simple.blade.php for better maintainability
 */

$(document).ready(function() {
    // ============================
    // PERFECT SCROLLBAR INITIALIZATION
    // ============================
    
    // Make PerfectScrollbar available globally if needed
    window.PerfectScrollbar = PerfectScrollbar;
    
    // Initialize PerfectScrollbar if not already initialized by app.js
    const scrollContainers = document.querySelectorAll('[data-perfect-scrollbar]');
    scrollContainers.forEach(container => {
        if (!container.classList.contains('ps')) {
            new PerfectScrollbar(container, {
                wheelSpeed: 2,
                wheelPropagation: true,
                minScrollbarLength: 20
            });
        }
    });

    // ============================
    // PROFILE DROPDOWN FUNCTIONALITY
    // ============================
    
    const $profileDropdown = $('.profile-dropdown');
    const $profileToggle = $('#profileDropdown');
    const $profileMenu = $('#profileMenu');

    // Toggle dropdown on click
    $profileToggle.off('click').on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();

        console.log('Profile dropdown toggled');
        $profileDropdown.toggleClass('active');

        // Prevent any other handlers from executing
        return false;
    });

    // Close dropdown when clicking outside
    $(document).off('click.profileDropdown').on('click.profileDropdown', function(e) {
        if (!$profileDropdown.is(e.target) && $profileDropdown.has(e.target).length === 0) {
            console.log('Closing profile dropdown - clicked outside');
            $profileDropdown.removeClass('active');
        }
    });

    // Prevent dropdown from closing when clicking inside
    $profileMenu.off('click').on('click', function(e) {
        console.log('Clicked inside profile menu');
        e.stopPropagation();
        return false;
    });

    // Close dropdown on escape key
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape') {
            $profileDropdown.removeClass('active');
        }
    });
    
    // ============================
    // HEADER POSITIONING FIXES
    // ============================
    
    // Ensure header stays fixed
    const header = document.querySelector('.mdk-header[data-fixed]');
    if (header) {
        // Force fixed positioning
        header.style.position = 'fixed';
        header.style.top = '0';
        header.style.left = '0';
        header.style.right = '0';
        header.style.width = '100%';
        header.style.zIndex = '1030';
        
        // Ensure content has proper spacing
        const content = document.querySelector('.mdk-header-layout__content');
        if (content) {
            content.style.paddingTop = '80px';
        }
    }
    
    // ============================
    // SIDEBAR MENU FUNCTIONALITY
    // ============================

    // Initialize sidebar menu state
    $('.sidebar-submenu.show').each(function() {
        $(this).closest('.sidebar-menu-item').addClass('open');
    });

    // Handle sidebar menu collapsing
    $('[data-toggle="collapse"]').off('click.sidebar').on('click.sidebar', function(e) {
        e.preventDefault();
        e.stopPropagation();

        const target = $(this).attr('href');
        const $target = $(target);
        const $parent = $(this).closest('.sidebar-menu-item');

        console.log('Sidebar toggle clicked', target);

        // Check if target exists
        if ($target.length === 0) {
            console.error('Collapse target not found:', target);
            return false;
        }

        // Toggle the submenu
        if ($target.hasClass('show')) {
            $target.removeClass('show');
            $parent.removeClass('open');
            console.log('Closing sidebar submenu:', target);
        } else {
            $target.addClass('show');
            $parent.addClass('open');
            console.log('Opening sidebar submenu:', target);
        }

        return false;
    });

    // Alternative click handler in case the first one doesn't work
    $('.sidebar-menu-button[data-toggle="collapse"]').off('click.sidebar').on('click.sidebar', function(e) {
        e.preventDefault();
        e.stopPropagation();

        const targetId = $(this).attr('href');
        const $target = $(targetId);
        const $parent = $(this).closest('.sidebar-menu-item');

        console.log('Alternative handler triggered for:', targetId);

        if ($target.hasClass('show')) {
            $target.removeClass('show');
            $parent.removeClass('open');
        } else {
            $target.addClass('show');
            $parent.addClass('open');
        }
    });

    // Simple fallback using plain event delegation
    $(document).on('click', '.sidebar-menu-button[href="#items_menu"]', function(e) {
        e.preventDefault();
        e.stopPropagation();

        console.log('Fallback handler for items menu triggered');

        const $submenu = $('#items_menu');
        const $parent = $(this).closest('.sidebar-menu-item');

        if ($submenu.hasClass('show')) {
            $submenu.removeClass('show');
            $parent.removeClass('open');
            console.log('Closing items submenu');
        } else {
            $submenu.addClass('show');
            $parent.addClass('open');
            console.log('Opening items submenu');
        }
    });

    // Projects menu fallback handler
    $(document).on('click', '.sidebar-menu-button[href="#projects_menu"]', function(e) {
        e.preventDefault();
        e.stopPropagation();

        console.log('Fallback handler for projects menu triggered');

        const $submenu = $('#projects_menu');
        const $parent = $(this).closest('.sidebar-menu-item');

        if ($submenu.hasClass('show')) {
            $submenu.removeClass('show');
            $parent.removeClass('open');
            console.log('Closing projects submenu');
        } else {
            $submenu.addClass('show');
            $parent.addClass('open');
            console.log('Opening projects submenu');
        }
    });

    // Inventory menu fallback handler
    $(document).on('click', '.sidebar-menu-button[href="#inventory_menu"]', function(e) {
        e.preventDefault();
        e.stopPropagation();

        console.log('Fallback handler for inventory menu triggered');

        const $submenu = $('#inventory_menu');
        const $parent = $(this).closest('.sidebar-menu-item');

        if ($submenu.hasClass('show')) {
            $submenu.removeClass('show');
            $parent.removeClass('open');
            console.log('Closing inventory submenu');
        } else {
            $submenu.addClass('show');
            $parent.addClass('open');
            console.log('Opening inventory submenu');
        }
    });

    // Prevent submenu area clicks from bubbling up
    $(document).on('click', '.sidebar-submenu', function(e) {
        e.stopPropagation();
        console.log('Clicked in submenu area');
    });

    // Handle clicks on submenu links properly
    $(document).on('click', '.sidebar-submenu .sidebar-menu-button', function(e) {
        // Don't prevent default here as we want navigation to work
        e.stopPropagation();
        console.log('Clicked submenu link:', $(this).attr('href'));
    });
    
    // ============================
    // PAGE LOADING IMPLEMENTATION
    // ============================
    
    const $loadingOverlay = $('#pageLoadingOverlay');
    const $pageContainer = $('.page__container');
    
    // Show loading on navigation links
    function showPageLoading() {
        $loadingOverlay.addClass('show');
        $pageContainer.addClass('loading');
    }
    
    // Hide loading overlay
    function hidePageLoading() {
        $loadingOverlay.removeClass('show');
        $pageContainer.removeClass('loading');
    }
    
    // Handle navigation clicks
    $(document).on('click', 'a[href]:not([href^="#"]):not([href^="javascript:"]):not([href^="mailto:"]):not([href^="tel:"]):not([data-toggle]):not([onclick]):not(.no-loading)', function(e) {
        const href = $(this).attr('href');
        
        // Skip if it's the current page or external link
        if (href === window.location.pathname || 
            href.startsWith('http') && !href.includes(window.location.hostname)) {
            return;
        }
        
        // Skip if it's a form submission or AJAX request
        if ($(this).closest('form').length > 0) {
            return;
        }
        
        // Show loading for internal navigation
        showPageLoading();
        
        // Hide loading after a timeout as fallback
        setTimeout(hidePageLoading, 5000);
    });
    
    // Handle form submissions that navigate
    $(document).on('submit', 'form:not(.ajax-form)', function() {
        const action = $(this).attr('action');
        const method = $(this).attr('method') || 'GET';
        
        // Only show loading for GET forms or forms that navigate
        if (method.toLowerCase() === 'get' || $(this).hasClass('navigate-form')) {
            showPageLoading();
            setTimeout(hidePageLoading, 5000);
        }
    });
    
    // Handle browser back/forward buttons
    $(window).on('popstate', function() {
        showPageLoading();
        setTimeout(hidePageLoading, 3000);
    });
    
    // Hide loading when page is fully loaded
    $(window).on('load', function() {
        hidePageLoading();
    });
    
    // Hide loading on page show (back button)
    $(window).on('pageshow', function() {
        hidePageLoading();
    });
    
    // ============================
    // TABLE DROPDOWN POSITIONING
    // ============================
    
    // Fix dropdown positioning for table rows
    $('.dropdown-toggle').on('click', function() {
        const $dropdown = $(this).closest('.dropdown');
        const $menu = $dropdown.find('.dropdown-menu');
        const $table = $dropdown.closest('table');
        const $row = $dropdown.closest('tr');
        
        // Check if this is one of the last two rows
        const rowIndex = $row.index();
        const totalRows = $table.find('tbody tr').length;
        
        if (rowIndex >= totalRows - 2) {
            $dropdown.addClass('dropup');
        } else {
            $dropdown.removeClass('dropup');
        }
    });
    
    // ============================
    // CLEANUP
    // ============================
    
    // Hide any unwanted app settings
    $('#app-settings, .app-settings').remove();
});

// ============================
// GLOBAL UTILITY FUNCTIONS
// ============================

/**
 * Show toast notification
 * @param {string} message - The message to display
 * @param {string} type - Type of alert: 'success' or 'error'
 */
function showToastMessage(message, type = 'success') {
    const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
    const toast = `<div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
        ${message}
        <button type="button" class="close" data-dismiss="alert">&times;</button>
    </div>`;
    $('body').append(toast);
    setTimeout(() => $('.alert').alert('close'), 5000);
}