// Modern Video Player Controller
class ModernVideoPlayer {
    constructor() {
        this.player = null;
        this.isPlaying = false;
        this.isMuted = true;
        this.duration = 0;
        this.currentTime = 0;
        this.volume = 1;
        
        this.init();
    }
    
    init() {
        // Wait for DOM to be ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.setup());
        } else {
            this.setup();
        }
    }
    
    setup() {
        this.loadYouTubeAPI();
        this.setupEventListeners();
        this.initAnimations();
    }
    
    loadYouTubeAPI() {
        if (typeof YT === 'undefined') {
            const tag = document.createElement('script');
            tag.src = 'https://www.youtube.com/iframe_api';
            const firstScriptTag = document.getElementsByTagName('script')[0];
            firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
        }
        
        window.onYouTubeIframeAPIReady = () => {
            this.initYouTubePlayer();
        };
    }
    
    initYouTubePlayer() {
        this.player = new YT.Player('ufg-player', {
            events: {
                'onReady': (event) => this.onPlayerReady(event),
                'onStateChange': (event) => this.onPlayerStateChange(event)
            }
        });
    }
    
    onPlayerReady(event) {
        console.log('Modern Video Player Ready');
        this.duration = this.player.getDuration();
        this.updateDuration();
        
        // Start playing
        this.player.mute();
        this.player.playVideo();
        this.isPlaying = true;
        this.isMuted = true;
        this.updatePlayButton();
        this.startProgressTracking();
    }
    
    onPlayerStateChange(event) {
        if (event.data === YT.PlayerState.PLAYING) {
            this.isPlaying = true;
            this.startProgressTracking();
        } else if (event.data === YT.PlayerState.PAUSED) {
            this.isPlaying = false;
            this.stopProgressTracking();
        } else if (event.data === YT.PlayerState.ENDED) {
            this.player.seekTo(0);
            this.player.playVideo();
        }
        this.updatePlayButton();
    }
    
    setupEventListeners() {
        // Play/Pause
        const playPauseBtn = document.querySelector('.play-pause');
        if (playPauseBtn) {
            playPauseBtn.addEventListener('click', () => this.togglePlay());
        }
        
        // Progress Bar
        const progressContainer = document.querySelector('.progress-container');
        if (progressContainer) {
            progressContainer.addEventListener('click', (e) => this.seekTo(e));
            progressContainer.addEventListener('mousemove', (e) => this.showSeekPreview(e));
        }
        
        // Volume
        const volumeBtn = document.querySelector('.volume-btn');
        if (volumeBtn) {
            volumeBtn.addEventListener('click', () => this.toggleMute());
        }
        
        const volumeTrack = document.querySelector('.volume-track');
        if (volumeTrack) {
            volumeTrack.addEventListener('click', (e) => this.setVolume(e));
        }
        
        // Fullscreen
        const fullscreenBtn = document.querySelector('.fullscreen-btn');
        if (fullscreenBtn) {
            fullscreenBtn.addEventListener('click', () => this.toggleFullscreen());
        }
        
        // Keyboard shortcuts
        document.addEventListener('keydown', (e) => this.handleKeyboard(e));
    }
    
    togglePlay() {
        if (!this.player) return;
        
        if (this.isPlaying) {
            this.player.pauseVideo();
        } else {
            this.player.playVideo();
        }
        
        this.addRippleEffect(event.currentTarget);
    }
    
    updatePlayButton() {
        const playIcon = document.querySelector('.play-icon');
        const pauseIcon = document.querySelector('.pause-icon');
        
        if (playIcon && pauseIcon) {
            if (this.isPlaying) {
                playIcon.style.display = 'none';
                pauseIcon.style.display = 'block';
            } else {
                playIcon.style.display = 'block';
                pauseIcon.style.display = 'none';
            }
        }
    }
    
    seekTo(event) {
        if (!this.player || !this.duration) return;
        
        const progressContainer = event.currentTarget;
        const rect = progressContainer.getBoundingClientRect();
        const clickX = event.clientX - rect.left;
        const percentage = clickX / rect.width;
        const seekTime = this.duration * percentage;
        
        this.player.seekTo(seekTime, true);
        this.updateProgress();
    }
    
    showSeekPreview(event) {
        if (!this.duration) return;
        
        const progressContainer = event.currentTarget;
        const rect = progressContainer.getBoundingClientRect();
        const hoverX = event.clientX - rect.left;
        const percentage = hoverX / rect.width;
        const previewTime = this.duration * percentage;
        
        // Update handle position for preview
        const handle = document.querySelector('.progress-handle');
        if (handle) {
            handle.style.left = `${percentage * 100}%`;
        }
    }
    
    toggleMute() {
        if (!this.player) return;
        
        if (this.isMuted) {
            this.player.unMute();
            this.player.setVolume(this.volume * 100);
            this.isMuted = false;
        } else {
            this.player.mute();
            this.isMuted = true;
        }
        
        this.updateVolumeButton();
        this.addRippleEffect(event.currentTarget);
    }
    
    updateVolumeButton() {
        const volumeHigh = document.querySelector('.volume-high');
        const volumeMuted = document.querySelector('.volume-muted');
        const volumeLevel = document.querySelector('.volume-level');
        
        if (volumeHigh && volumeMuted) {
            if (this.isMuted) {
                volumeHigh.style.display = 'none';
                volumeMuted.style.display = 'block';
                if (volumeLevel) volumeLevel.style.width = '0%';
            } else {
                volumeHigh.style.display = 'block';
                volumeMuted.style.display = 'none';
                if (volumeLevel) volumeLevel.style.width = `${this.volume * 100}%`;
            }
        }
    }
    
    setVolume(event) {
        if (!this.player) return;
        
        const volumeTrack = event.currentTarget;
        const rect = volumeTrack.getBoundingClientRect();
        const clickX = event.clientX - rect.left;
        const percentage = Math.max(0, Math.min(1, clickX / rect.width));
        
        this.volume = percentage;
        this.player.setVolume(percentage * 100);
        
        if (percentage > 0 && this.isMuted) {
            this.player.unMute();
            this.isMuted = false;
        } else if (percentage === 0) {
            this.player.mute();
            this.isMuted = true;
        }
        
        this.updateVolumeButton();
    }
    
    toggleFullscreen() {
        const videoFrame = document.querySelector('.video-frame');
        if (!videoFrame) return;
        
        if (!document.fullscreenElement) {
            videoFrame.requestFullscreen().catch(err => {
                console.error('Error attempting to enable fullscreen:', err);
            });
        } else {
            document.exitFullscreen();
        }
        
        this.addRippleEffect(event.currentTarget);
    }
    
    handleKeyboard(event) {
        if (!this.player) return;
        
        switch(event.key) {
            case ' ':
            case 'k':
                event.preventDefault();
                this.togglePlay();
                break;
            case 'ArrowLeft':
                event.preventDefault();
                this.player.seekTo(this.currentTime - 5, true);
                break;
            case 'ArrowRight':
                event.preventDefault();
                this.player.seekTo(this.currentTime + 5, true);
                break;
            case 'ArrowUp':
                event.preventDefault();
                this.setVolumeByKey(0.1);
                break;
            case 'ArrowDown':
                event.preventDefault();
                this.setVolumeByKey(-0.1);
                break;
            case 'm':
                event.preventDefault();
                this.toggleMute();
                break;
            case 'f':
                event.preventDefault();
                this.toggleFullscreen();
                break;
        }
    }
    
    setVolumeByKey(change) {
        this.volume = Math.max(0, Math.min(1, this.volume + change));
        this.player.setVolume(this.volume * 100);
        
        if (this.volume > 0 && this.isMuted) {
            this.player.unMute();
            this.isMuted = false;
        }
        
        this.updateVolumeButton();
    }
    
    startProgressTracking() {
        this.stopProgressTracking();
        this.progressInterval = setInterval(() => this.updateProgress(), 100);
    }
    
    stopProgressTracking() {
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
            this.progressInterval = null;
        }
    }
    
    updateProgress() {
        if (!this.player) return;
        
        try {
            this.currentTime = this.player.getCurrentTime();
            this.duration = this.player.getDuration();
            
            const percentage = this.duration > 0 ? (this.currentTime / this.duration) * 100 : 0;
            
            // Update progress bar
            const progressPlayed = document.querySelector('.progress-played');
            const progressHandle = document.querySelector('.progress-handle');
            
            if (progressPlayed) {
                progressPlayed.style.width = `${percentage}%`;
            }
            
            if (progressHandle) {
                progressHandle.style.left = `${percentage}%`;
            }
            
            // Update buffered
            const bufferedPercentage = this.player.getVideoLoadedFraction() * 100;
            const progressBuffered = document.querySelector('.progress-buffered');
            if (progressBuffered) {
                progressBuffered.style.width = `${bufferedPercentage}%`;
            }
            
            // Update time display
            this.updateTimeDisplay();
        } catch (error) {
            console.error('Error updating progress:', error);
        }
    }
    
    updateTimeDisplay() {
        const currentTimeEl = document.querySelector('.current-time');
        const durationEl = document.querySelector('.duration');
        
        if (currentTimeEl) {
            currentTimeEl.textContent = this.formatTime(this.currentTime);
        }
        
        if (durationEl) {
            durationEl.textContent = this.formatTime(this.duration);
        }
    }
    
    updateDuration() {
        const durationEl = document.querySelector('.duration');
        if (durationEl) {
            durationEl.textContent = this.formatTime(this.duration);
        }
    }
    
    formatTime(seconds) {
        if (!seconds || isNaN(seconds)) return '0:00';
        
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const remainingSeconds = Math.floor(seconds % 60);
        
        if (hours > 0) {
            return `${hours}:${minutes.toString().padStart(2, '0')}:${remainingSeconds.toString().padStart(2, '0')}`;
        }
        
        return `${minutes}:${remainingSeconds.toString().padStart(2, '0')}`;
    }
    
    addRippleEffect(element) {
        const ripple = document.createElement('span');
        ripple.classList.add('ripple');
        element.appendChild(ripple);
        
        setTimeout(() => {
            ripple.remove();
        }, 600);
    }
    
    initAnimations() {
        // Add loading class initially
        const modernPlayer = document.querySelector('.modern-player');
        if (modernPlayer) {
            modernPlayer.classList.add('loading');
            
            // Remove loading class when player is ready
            setTimeout(() => {
                modernPlayer.classList.remove('loading');
            }, 2000);
        }
    }
}

// Initialize the modern video player
const modernPlayer = new ModernVideoPlayer();

// Export for external access
window.ModernVideoPlayer = modernPlayer;