// Product Filter Functionality - Updated for new ARALUX design
document.addEventListener('DOMContentLoaded', function() {
    console.log('Setting up product filters...');
    
    // Filter buttons
    const categoryFilters = document.querySelectorAll('.category-filter');
    const standardsFilters = document.querySelectorAll('.standards-filter');
    const applicationFilters = document.querySelectorAll('.application-filter');
    
    // Products container
    const productsGrid = document.getElementById('products-grid');
    
    if (!productsGrid) {
        console.log('Products grid not found');
        return;
    }
    
    // Get all product cards
    const productCards = productsGrid.querySelectorAll('.product-card');
    console.log(`Found ${productCards.length} product cards`);
    
    // Current active filters
    let activeFilters = {
        category: 'all',
        standard: 'all',
        application: 'all'
    };
    
    // Set active filter button
    function setActiveFilter(buttons, activeButton) {
        buttons.forEach(btn => {
            btn.classList.remove('active');
            btn.classList.remove('bg-aufg-gradient', 'text-white');
            btn.classList.add('bg-gray-100', 'text-aufg-text-darker');
        });
        
        activeButton.classList.add('active');
        activeButton.classList.remove('bg-gray-100', 'text-aufg-text-darker');
        activeButton.classList.add('bg-aufg-gradient', 'text-white');
    }
    
    // Filter products based on active filters
    function filterProducts() {
        console.log('Filtering products with:', activeFilters);
        
        let visibleCount = 0;
        
        productCards.forEach(card => {
            let showCard = true;
            
            // Category filter
            if (activeFilters.category !== 'all') {
                const cardCategory = card.getAttribute('data-category');
                if (cardCategory !== activeFilters.category) {
                    showCard = false;
                }
            }
            
            // Standards filter
            if (activeFilters.standard !== 'all' && showCard) {
                const cardStandards = card.getAttribute('data-standard');
                if (!cardStandards || !cardStandards.includes(activeFilters.standard)) {
                    showCard = false;
                }
            }
            
            // Applications filter
            if (activeFilters.application !== 'all' && showCard) {
                const cardApplications = card.getAttribute('data-application');
                if (!cardApplications || !cardApplications.includes(activeFilters.application)) {
                    showCard = false;
                }
            }
            
            // Show/hide card with animation
            if (showCard) {
                card.style.display = 'block';
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                setTimeout(() => {
                    card.style.opacity = '1';
                    card.style.transform = 'translateY(0)';
                    card.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
                }, 50);
                visibleCount++;
            } else {
                card.style.display = 'none';
            }
        });
        
        console.log(`${visibleCount} products visible after filtering`);
        
        // Show "no results" message if needed
        showNoResultsMessage(visibleCount === 0);
    }
    
    // Show/hide no results message
    function showNoResultsMessage(show) {
        let noResultsMsg = document.querySelector('.no-results-message');
        
        if (show && !noResultsMsg) {
            noResultsMsg = document.createElement('div');
            noResultsMsg.className = 'no-results-message col-span-full text-center py-12';
            noResultsMsg.innerHTML = `
                <div class="text-aufg-text-secondary">
                    <svg class="w-16 h-16 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M9.172 16.172a4 4 0 015.656 0M9 12h6m-6-4h6m2 5.291A7.962 7.962 0 0112 15c-2.34 0-4.29-1.009-5.824-2.562M15.71 9.71L21 4.29M21 21l-5.657-5.657M5.343 5.343L1 1"/>
                    </svg>
                    <h3 class="text-lg font-medium mb-2">No products found</h3>
                    <p class="text-sm">Try adjusting your filters to see more results</p>
                </div>
            `;
            productsGrid.appendChild(noResultsMsg);
        } else if (!show && noResultsMsg) {
            noResultsMsg.remove();
        }
    }
    
    // Category filter event listeners
    categoryFilters.forEach(button => {
        button.addEventListener('click', function() {
            const category = this.getAttribute('data-category');
            activeFilters.category = category;
            setActiveFilter(categoryFilters, this);
            filterProducts();
            console.log(`Category filter changed to: ${category}`);
        });
    });
    
    // Standards filter event listeners
    standardsFilters.forEach(button => {
        button.addEventListener('click', function() {
            const standard = this.getAttribute('data-standard');
            activeFilters.standard = standard;
            setActiveFilter(standardsFilters, this);
            filterProducts();
            console.log(`Standards filter changed to: ${standard}`);
        });
    });
    
    // Applications filter event listeners
    applicationFilters.forEach(button => {
        button.addEventListener('click', function() {
            const application = this.getAttribute('data-application');
            activeFilters.application = application;
            setActiveFilter(applicationFilters, this);
            filterProducts();
            console.log(`Applications filter changed to: ${application}`);
        });
    });
    
    // Technical Data Sheet and Request Quote button functionality
    // Note: querySelectorAll doesn't support :contains, so we'll use a different approach
    const allButtons = document.querySelectorAll('#products-grid button');
    
    allButtons.forEach(button => {
        if (button.textContent.includes('Technical Data Sheet')) {
            button.addEventListener('click', function() {
                // Get product info from the parent card
                const productCard = this.closest('.product-card');
                const productName = productCard.querySelector('h3').textContent;
                
                // For now, show an alert - in production this would download/open the PDF
                alert(`Technical Data Sheet for ${productName} would be downloaded here.`);
                console.log(`Technical data sheet requested for: ${productName}`);
            });
        } else if (button.textContent.includes('General Inquiry')) {
            button.addEventListener('click', function() {
                // Get product info from the parent card
                const productCard = this.closest('.product-card');
                const productName = productCard.querySelector('h3').textContent;
                const productImage = productCard.querySelector('img').src;
                const productDescription = productCard.querySelector('p:nth-of-type(2)').textContent;
                
                // Open modal with product details
                openQuoteModal();
                console.log(`General inquiry requested for: ${productName}`);
            });
        }
    });
    
    // In-page Product Detail functionality
    function showProductDetail(productCard) {
        const detailContainer = document.getElementById('product-detail');
        if (!detailContainer) return;
        
        const productData = {
            name: productCard.getAttribute('data-product-name'),
            image: productCard.getAttribute('data-product-image'),
            description: productCard.getAttribute('data-product-description'),
            thickness: productCard.getAttribute('data-thickness'),
            lightTransmission: productCard.getAttribute('data-light-transmission'),
            maxSize: productCard.getAttribute('data-max-size'),
            applications: productCard.getAttribute('data-applications'),
            price: productCard.getAttribute('data-price')
        };
        
        // Populate detail container with content
        document.getElementById('detail-image').src = productData.image;
        document.getElementById('detail-title').textContent = productData.name;
        document.getElementById('detail-subtitle').textContent = `Thickness: ${productData.thickness}`;
        document.getElementById('detail-description').textContent = productData.description;
        
        // Populate applications
        const applicationsContainer = document.getElementById('detail-applications');
        const applications = productData.applications.split(', ');
        applicationsContainer.innerHTML = applications.map(app => 
            `<span class="bg-blue-50 text-blue-700 px-3 py-1 rounded-full text-sm font-medium">${app.trim()}</span>`
        ).join('');
        
        // Populate technical specs
        const specsContainer = document.getElementById('detail-specs');
        specsContainer.innerHTML = `
            <div class="flex justify-between items-center">
                <span class="text-aufg-text-secondary">Light Transmission</span>
                <span class="font-semibold text-aufg-primary-darker">${productData.lightTransmission}</span>
            </div>
            <div class="flex justify-between items-center">
                <span class="text-aufg-text-secondary">Maximum Size</span>
                <span class="font-semibold text-aufg-primary-darker">${productData.maxSize}</span>
            </div>
            <div class="flex justify-between items-center">
                <span class="text-aufg-text-secondary">Thickness Range</span>
                <span class="font-semibold text-aufg-primary-darker">${productData.thickness}</span>
            </div>
            <div class="flex justify-between items-center">
                <span class="text-aufg-text-secondary">Pricing</span>
                <span class="font-semibold text-aufg-primary-darker">${productData.price}</span>
            </div>
        `;
        
        // Show detail container with smooth animation
        detailContainer.classList.remove('hidden');
        setTimeout(() => {
            detailContainer.classList.remove('opacity-0');
            detailContainer.classList.add('opacity-100');
            
            // Smooth scroll to detail container
            detailContainer.scrollIntoView({ 
                behavior: 'smooth', 
                block: 'start' 
            });
        }, 10);
    }
    
    // Close detail functionality
    function closeProductDetail() {
        const detailContainer = document.getElementById('product-detail');
        if (!detailContainer) return;
        
        detailContainer.classList.remove('opacity-100');
        detailContainer.classList.add('opacity-0');
        
        setTimeout(() => {
            detailContainer.classList.add('hidden');
        }, 300);
    }
    
    // Quote modal functionality
    function openQuoteModal() {
        const modal = document.getElementById('quote-modal');
        if (!modal) return;
        
        // Show modal with animations
        modal.classList.remove('hidden');
        setTimeout(() => {
            modal.classList.remove('opacity-0');
            modal.querySelector('.transform').classList.remove('scale-95');
            modal.querySelector('.transform').classList.add('scale-100');
        }, 10);
        
        document.body.style.overflow = 'hidden';
    }
    
    function closeQuoteModal() {
        const modal = document.getElementById('quote-modal');
        if (modal) {
            modal.classList.add('opacity-0');
            modal.querySelector('.transform').classList.remove('scale-100');
            modal.querySelector('.transform').classList.add('scale-95');
            setTimeout(() => {
                modal.classList.add('hidden');
                document.body.style.overflow = 'auto';
            }, 300);
        }
    }
    
    // Event listeners for close buttons
    document.addEventListener('click', function(e) {
        // Quote modal close
        if (e.target.id === 'close-quote-modal' || e.target.closest('#close-quote-modal')) {
            closeQuoteModal();
        }
        
        // Close on backdrop click
        if (e.target.id === 'quote-modal') {
            closeQuoteModal();
        }
    });
    
    // Keyboard escape to close modals
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeQuoteModal();
        }
    });
    
    // Close detail event listener
    const closeDetailBtn = document.getElementById('close-detail');
    if (closeDetailBtn) {
        closeDetailBtn.addEventListener('click', closeProductDetail);
    }
    
    // Product thickness mapping
    const productThickness = {
        'decorative-mirror': ['3mm', '4mm', '5mm', '6mm'],
        'clear-float': ['3mm', '4mm', '5mm', '6mm', '8mm', '10mm', '12mm'],
        'pattern-glass': ['4mm', '5mm', '6mm', '8mm'],
        'silver-mirror': ['3mm', '4mm', '5mm', '6mm'],
        'laminated-glass': ['6mm', '8mm', '10mm', '12mm'],
        'tempered-glass': ['4mm', '5mm', '6mm', '8mm', '10mm', '12mm']
    };

    // Handle product selection change
    document.addEventListener('change', function(e) {
        if (e.target.id === 'product-select') {
            const productValue = e.target.value;
            const thicknessSelect = document.getElementById('thickness-select');
            
            if (productValue && productThickness[productValue]) {
                // Enable thickness dropdown and populate options
                thicknessSelect.disabled = false;
                thicknessSelect.innerHTML = '<option value="">Select Thickness</option>';
                
                productThickness[productValue].forEach(thickness => {
                    const option = document.createElement('option');
                    option.value = thickness;
                    option.textContent = thickness;
                    thicknessSelect.appendChild(option);
                });
            } else {
                // Disable thickness dropdown
                thicknessSelect.disabled = true;
                thicknessSelect.innerHTML = '<option value="">Select Product First</option>';
            }
        }
    });

    // Make functions global for onclick handlers
    window.showProductDetail = showProductDetail;
    window.openQuoteModal = openQuoteModal;
    window.closeProductDetail = closeProductDetail;
    
    // Quotation form submission
    const quotationForm = document.getElementById('quotation-form');
    if (quotationForm) {
        quotationForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // Get form data
            const formData = new FormData(this);
            const productName = document.getElementById('modal-title').textContent;
            
            // For now, show success message - in production this would send to server
            alert(`Thank you! Your general inquiry for ${productName} has been submitted. We will contact you soon.`);
            
            // Close modal
            closeQuoteModal();
            
            // Reset form
            this.reset();
            
            console.log('General inquiry form submitted for:', productName);
        });
    }
    
    // Initialize with default filters
    filterProducts();
    
    console.log('Product filters initialized successfully');
});