// Products Horizontal Scroll Functionality
document.addEventListener('DOMContentLoaded', function() {
    console.log('Setting up products horizontal scroll...');
    
    const carousel = document.querySelector('.products-carousel');
    if (!carousel) {
        console.log('Products container not found');
        return;
    }
    
    // Add new product cards
    const existingCards = carousel.querySelectorAll('.product-card');
    console.log(`Found ${existingCards.length} existing cards`);
    
    // Check if Decorative Mirror card exists (it should be in HTML now)
    const decorativeMirror = carousel.querySelector('.product-card-decorative-mirror');
    if (decorativeMirror) {
        console.log('Decorative Mirror card found in HTML');
    } else {
        console.log('Warning: Decorative Mirror card not found - check HTML structure');
    }
    
    // Get all product cards
    const cards = carousel.querySelectorAll('.product-card');
    const totalCards = cards.length;
    
    if (totalCards === 0) {
        console.log('No product cards found');
        return;
    }
    
    console.log(`Found ${totalCards} product cards`);
    
    // Check if wrapper already exists (to avoid duplicate wrapping)
    let carouselWrapper = carousel.closest('.products-carousel-wrapper');
    
    if (!carouselWrapper) {
        // Create wrapper for horizontal scroll
        carouselWrapper = document.createElement('div');
        carouselWrapper.className = 'products-carousel-wrapper';
        
        // Move existing container into the wrapper
        carousel.parentNode.insertBefore(carouselWrapper, carousel);
        carouselWrapper.appendChild(carousel);
    }
    
    // Add navigation buttons (only if they don't exist)
    if (!carouselWrapper.querySelector('.carousel-nav')) {
        const navHTML = `
            <button class="carousel-nav carousel-prev" aria-label="Scroll left">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none">
                    <path d="M15 18L9 12L15 6" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>
            </button>
            <button class="carousel-nav carousel-next" aria-label="Scroll right">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none">
                    <path d="M9 18L15 12L9 6" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>
            </button>
        `;
        
        carouselWrapper.insertAdjacentHTML('beforeend', navHTML);
    }
    
    // No dots needed for horizontal scroll
    
    // Get control elements
    const prevBtn = carouselWrapper.querySelector('.carousel-prev');
    const nextBtn = carouselWrapper.querySelector('.carousel-next');
    
    // Horizontal scroll functionality
    const scrollAmount = 320; // Card width + gap
    
    // Scroll functions
    function scrollLeft() {
        carousel.scrollBy({
            left: -scrollAmount,
            behavior: 'smooth'
        });
        console.log('Scrolling left');
    }
    
    function scrollRight() {
        carousel.scrollBy({
            left: scrollAmount,
            behavior: 'smooth'
        });
        console.log('Scrolling right');
    }
    
    // Update navigation button states
    function updateNavButtons() {
        const scrollLeft = carousel.scrollLeft;
        const scrollWidth = carousel.scrollWidth;
        const clientWidth = carousel.clientWidth;
        
        // Update button opacity based on scroll position
        prevBtn.style.opacity = scrollLeft <= 0 ? '0.5' : '1';
        nextBtn.style.opacity = scrollLeft >= scrollWidth - clientWidth - 10 ? '0.5' : '1';
    }
    
    // Card fade effects disabled to prevent hover conflicts
    function updateCardFadeEffects() {
        // Disabled fade effects to prevent hover animation conflicts
        // Cards will maintain consistent visibility without fade classes
        console.log('Card fade effects disabled for smooth hover performance');
    }
    
    // Initial button state and card fade update
    updateNavButtons();
    updateCardFadeEffects();
    
    // Event listeners
    nextBtn.addEventListener('click', (e) => {
        e.preventDefault();
        scrollRight();
    });
    
    prevBtn.addEventListener('click', (e) => {
        e.preventDefault();
        scrollLeft();
    });
    
    // No dots to handle
    
    // Update navigation on scroll and apply card fade effects
    carousel.addEventListener('scroll', () => {
        updateNavButtons();
        updateCardFadeEffects();
    }, { passive: true });
    
    // Touch/swipe support for horizontal scrolling
    let startX = 0;
    let scrollLeftStart = 0;
    let isScrolling = false;
    
    carousel.addEventListener('touchstart', (e) => {
        startX = e.touches[0].clientX;
        scrollLeftStart = carousel.scrollLeft;
        isScrolling = false;
    }, { passive: true });
    
    carousel.addEventListener('touchmove', (e) => {
        if (!isScrolling) {
            const currentX = e.touches[0].clientX;
            const deltaX = startX - currentX;
            carousel.scrollLeft = scrollLeftStart + deltaX;
            isScrolling = true;
        }
    }, { passive: true });
    
    carousel.addEventListener('touchend', () => {
        isScrolling = false;
        updateNavButtons();
        updateCardFadeEffects();
    }, { passive: true });
    
    // Keyboard navigation
    carouselWrapper.addEventListener('keydown', (e) => {
        if (e.key === 'ArrowLeft') {
            e.preventDefault();
            scrollLeft();
        } else if (e.key === 'ArrowRight') {
            e.preventDefault();
            scrollRight();
        }
    });
    
    // Mouse wheel horizontal scrolling
    carousel.addEventListener('wheel', (e) => {
        if (Math.abs(e.deltaX) > Math.abs(e.deltaY)) {
            // Horizontal wheel scroll - let it work naturally
            return;
        }
        
        // Vertical wheel scroll - convert to horizontal
        e.preventDefault();
        carousel.scrollBy({
            left: e.deltaY > 0 ? scrollAmount / 2 : -scrollAmount / 2,
            behavior: 'smooth'
        });
    }, { passive: false });
    
    // Handle window resize
    window.addEventListener('resize', () => {
        updateNavButtons();
        updateCardFadeEffects();
    });
    
    // Make wrapper focusable for keyboard navigation
    carouselWrapper.setAttribute('tabindex', '0');
    
    console.log('Products horizontal scroll initialized successfully');
});