// Video Preloader - Advanced preloading functionality
class VideoPreloader {
    constructor() {
        this.preloadedVideos = new Map();
        this.preloadQueue = [];
        this.isPreloading = false;
        this.maxConcurrentPreloads = 2;
        this.currentPreloads = 0;
        
        // Performance metrics
        this.metrics = {
            totalPreloaded: 0,
            totalPreloadTime: 0,
            averagePreloadTime: 0,
            errors: 0
        };

        this.init();
    }

    init() {
        // Check if browser supports the features we need
        this.browserSupport = {
            intersectionObserver: 'IntersectionObserver' in window,
            connectionApi: 'connection' in navigator,
            localStorage: this.hasLocalStorage(),
            requestIdleCallback: 'requestIdleCallback' in window
        };

        // Start preloading during idle time
        this.scheduleIdlePreloading();
        
        // Monitor network conditions
        this.monitorNetworkConditions();
    }

    hasLocalStorage() {
        try {
            localStorage.setItem('test', 'test');
            localStorage.removeItem('test');
            return true;
        } catch (e) {
            return false;
        }
    }

    async preloadVideo(videoConfig, priority = 'normal') {
        const cacheKey = `${videoConfig.key}_${videoConfig.videoId}`;
        
        // Check if already preloaded
        if (this.preloadedVideos.has(cacheKey)) {
            return this.preloadedVideos.get(cacheKey);
        }

        // Add to queue based on priority
        const preloadTask = {
            config: videoConfig,
            cacheKey,
            priority,
            startTime: Date.now()
        };

        if (priority === 'high') {
            this.preloadQueue.unshift(preloadTask);
        } else {
            this.preloadQueue.push(preloadTask);
        }

        // Start processing queue
        this.processPreloadQueue();

        return new Promise((resolve, reject) => {
            preloadTask.resolve = resolve;
            preloadTask.reject = reject;
        });
    }

    async processPreloadQueue() {
        if (this.isPreloading || this.currentPreloads >= this.maxConcurrentPreloads) {
            return;
        }

        const task = this.preloadQueue.shift();
        if (!task) {
            return;
        }

        this.currentPreloads++;
        this.isPreloading = true;

        try {
            const result = await this.executePreload(task);
            this.preloadedVideos.set(task.cacheKey, result);
            
            // Update metrics
            const preloadTime = Date.now() - task.startTime;
            this.updateMetrics(preloadTime);

            // Cache success in localStorage
            this.cachePreloadSuccess(task.cacheKey, result);

            if (task.resolve) {
                task.resolve(result);
            }

            console.log(`✅ Preloaded video: ${task.config.key} in ${preloadTime}ms`);

        } catch (error) {
            this.metrics.errors++;
            console.error(`❌ Failed to preload video: ${task.config.key}`, error);
            
            if (task.reject) {
                task.reject(error);
            }
        } finally {
            this.currentPreloads--;
            
            // Continue processing queue
            setTimeout(() => {
                this.isPreloading = false;
                this.processPreloadQueue();
            }, 100);
        }
    }

    async executePreload(task) {
        const { config } = task;

        switch (config.type) {
            case 'youtube':
                return await this.preloadYouTubeVideo(config);
            case 'vimeo':
                return await this.preloadVimeoVideo(config);
            case 'mp4':
                return await this.preloadMP4Video(config);
            default:
                throw new Error(`Unsupported video type: ${config.type}`);
        }
    }

    async preloadYouTubeVideo(config) {
        const promises = [];

        // Preload thumbnail image
        if (config.videoId) {
            const thumbnailPromise = this.preloadImage(
                `https://img.youtube.com/vi/${config.videoId}/maxresdefault.jpg`
            );
            promises.push(thumbnailPromise);

            // Also preload alternative thumbnail sizes
            promises.push(
                this.preloadImage(`https://img.youtube.com/vi/${config.videoId}/hqdefault.jpg`)
            );
        }

        // Preload YouTube iframe API if not already loaded
        if (typeof YT === 'undefined') {
            promises.push(this.loadYouTubeAPI());
        }

        // On fast connections, preload some video data
        if (this.getConnectionSpeed() === 'fast') {
            promises.push(this.preloadYouTubeVideoData(config));
        }

        const results = await Promise.allSettled(promises);
        
        return {
            type: 'youtube',
            videoId: config.videoId,
            thumbnails: results.filter(r => r.status === 'fulfilled').map(r => r.value),
            preloadedAt: new Date().toISOString()
        };
    }

    async preloadVimeoVideo(config) {
        // Similar implementation for Vimeo
        const videoId = config.videoId;
        
        if (!videoId) {
            throw new Error('Vimeo video ID not found');
        }

        // Preload Vimeo thumbnail
        try {
            const response = await fetch(`https://vimeo.com/api/v2/video/${videoId}.json`);
            const data = await response.json();
            
            if (data && data[0] && data[0].thumbnail_large) {
                await this.preloadImage(data[0].thumbnail_large);
            }
        } catch (error) {
            console.warn('Could not preload Vimeo thumbnail:', error);
        }

        return {
            type: 'vimeo',
            videoId: config.videoId,
            preloadedAt: new Date().toISOString()
        };
    }

    async preloadMP4Video(config) {
        return new Promise((resolve, reject) => {
            const video = document.createElement('video');
            video.preload = 'metadata';
            video.muted = true;
            
            const timeout = setTimeout(() => {
                reject(new Error('MP4 preload timeout'));
            }, 10000);

            video.onloadedmetadata = () => {
                clearTimeout(timeout);
                resolve({
                    type: 'mp4',
                    url: config.videoUrl,
                    duration: video.duration,
                    preloadedAt: new Date().toISOString()
                });
            };

            video.onerror = () => {
                clearTimeout(timeout);
                reject(new Error('Failed to preload MP4 video'));
            };

            video.src = config.videoUrl || config.embedUrl;
        });
    }

    async preloadImage(url) {
        return new Promise((resolve, reject) => {
            const img = new Image();
            
            const timeout = setTimeout(() => {
                reject(new Error('Image preload timeout'));
            }, 5000);

            img.onload = () => {
                clearTimeout(timeout);
                resolve(url);
            };

            img.onerror = () => {
                clearTimeout(timeout);
                reject(new Error(`Failed to preload image: ${url}`));
            };

            img.src = url;
        });
    }

    async loadYouTubeAPI() {
        return new Promise((resolve, reject) => {
            // Check if already loaded
            if (typeof YT !== 'undefined' && YT.Player) {
                resolve();
                return;
            }

            // Check if script is already being loaded
            if (window.youtubeAPILoading) {
                // Wait for existing load
                const checkInterval = setInterval(() => {
                    if (typeof YT !== 'undefined' && YT.Player) {
                        clearInterval(checkInterval);
                        resolve();
                    }
                }, 100);
                return;
            }

            window.youtubeAPILoading = true;

            const script = document.createElement('script');
            script.src = 'https://www.youtube.com/iframe_api';
            script.async = true;

            script.onload = () => {
                window.youtubeAPILoading = false;
                resolve();
            };

            script.onerror = () => {
                window.youtubeAPILoading = false;
                reject(new Error('Failed to load YouTube API'));
            };

            document.head.appendChild(script);
        });
    }

    async preloadYouTubeVideoData(config) {
        // Create a hidden iframe to preload video data
        return new Promise((resolve) => {
            const iframe = document.createElement('iframe');
            iframe.src = config.embedUrl + '&autoplay=0&mute=1';
            iframe.style.display = 'none';
            iframe.style.position = 'absolute';
            iframe.style.left = '-9999px';
            iframe.width = '1';
            iframe.height = '1';

            iframe.onload = () => {
                // Remove after a short delay to allow some preloading
                setTimeout(() => {
                    if (iframe.parentNode) {
                        iframe.parentNode.removeChild(iframe);
                    }
                    resolve();
                }, 2000);
            };

            document.body.appendChild(iframe);
        });
    }

    scheduleIdlePreloading() {
        const scheduleNext = () => {
            if (this.browserSupport.requestIdleCallback) {
                requestIdleCallback(() => this.performIdlePreloading());
            } else {
                setTimeout(() => this.performIdlePreloading(), 1000);
            }
        };

        // Initial schedule
        scheduleNext();

        // Reschedule periodically
        setInterval(scheduleNext, 30000); // Every 30 seconds
    }

    async performIdlePreloading() {
        // Only preload during idle time and good network conditions
        if (this.currentPreloads > 0 || this.getConnectionSpeed() === 'slow') {
            return;
        }

        // Fetch latest video configs and preload any new videos
        try {
            const response = await fetch('/api/video-configs');
            const configs = await response.json();

            for (const [key, config] of Object.entries(configs)) {
                const cacheKey = `${config.key}_${config.videoId}`;
                
                if (!this.preloadedVideos.has(cacheKey) && config.preloadAt) {
                    await this.preloadVideo(config, 'low');
                }
            }
        } catch (error) {
            console.warn('Idle preloading failed:', error);
        }
    }

    monitorNetworkConditions() {
        if (this.browserSupport.connectionApi) {
            const connection = navigator.connection;
            
            // Adjust preloading based on connection type
            const updatePreloadStrategy = () => {
                switch (connection.effectiveType) {
                    case 'slow-2g':
                    case '2g':
                        this.maxConcurrentPreloads = 1;
                        break;
                    case '3g':
                        this.maxConcurrentPreloads = 2;
                        break;
                    case '4g':
                        this.maxConcurrentPreloads = 3;
                        break;
                    default:
                        this.maxConcurrentPreloads = 2;
                }
            };

            updatePreloadStrategy();
            connection.addEventListener('change', updatePreloadStrategy);
        }
    }

    getConnectionSpeed() {
        if (this.browserSupport.connectionApi) {
            const connection = navigator.connection;
            
            if (connection.effectiveType === '4g' || connection.downlink > 10) {
                return 'fast';
            } else if (connection.effectiveType === '3g' || connection.downlink > 1) {
                return 'medium';
            } else {
                return 'slow';
            }
        }
        
        // Default to medium if we can't detect
        return 'medium';
    }

    updateMetrics(preloadTime) {
        this.metrics.totalPreloaded++;
        this.metrics.totalPreloadTime += preloadTime;
        this.metrics.averagePreloadTime = this.metrics.totalPreloadTime / this.metrics.totalPreloaded;
    }

    cachePreloadSuccess(cacheKey, result) {
        if (this.browserSupport.localStorage) {
            try {
                const cacheData = {
                    result,
                    timestamp: Date.now(),
                    expiresAt: Date.now() + (24 * 60 * 60 * 1000) // 24 hours
                };
                
                localStorage.setItem(`video_preload_${cacheKey}`, JSON.stringify(cacheData));
            } catch (e) {
                // Storage quota exceeded or other error
                console.warn('Could not cache preload result:', e);
            }
        }
    }

    getCachedPreload(cacheKey) {
        if (this.browserSupport.localStorage) {
            try {
                const cached = localStorage.getItem(`video_preload_${cacheKey}`);
                if (cached) {
                    const data = JSON.parse(cached);
                    
                    if (Date.now() < data.expiresAt) {
                        return data.result;
                    } else {
                        // Expired, remove from cache
                        localStorage.removeItem(`video_preload_${cacheKey}`);
                    }
                }
            } catch (e) {
                console.warn('Could not read preload cache:', e);
            }
        }
        return null;
    }

    clearCache() {
        if (this.browserSupport.localStorage) {
            const keys = Object.keys(localStorage);
            keys.forEach(key => {
                if (key.startsWith('video_preload_')) {
                    localStorage.removeItem(key);
                }
            });
        }
        this.preloadedVideos.clear();
    }

    getMetrics() {
        return {
            ...this.metrics,
            browserSupport: this.browserSupport,
            connectionSpeed: this.getConnectionSpeed(),
            preloadedCount: this.preloadedVideos.size,
            queueLength: this.preloadQueue.length
        };
    }

    // Public API methods
    isVideoPreloaded(videoKey, videoId) {
        const cacheKey = `${videoKey}_${videoId}`;
        return this.preloadedVideos.has(cacheKey) || this.getCachedPreload(cacheKey) !== null;
    }

    forcePreload(videoConfig) {
        return this.preloadVideo(videoConfig, 'high');
    }
}

// Create global instance
window.videoPreloader = new VideoPreloader();

// Export for modules
if (typeof module !== 'undefined' && module.exports) {
    module.exports = VideoPreloader;
}