// Job Application Modal React Component
const { useState, useEffect } = React;

function JobApplicationModal({ isOpen, onClose, jobId, jobTitle }) {
    const [formData, setFormData] = useState({
        first_name: '',
        last_name: '',
        email: '',
        phone: '',
        cover_letter: '',
        resume: null
    });
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [toast, setToast] = useState({ show: false, message: '', type: 'success' });

    // Reset form when modal opens
    useEffect(() => {
        if (isOpen) {
            setFormData({
                first_name: '',
                last_name: '',
                email: '',
                phone: '',
                cover_letter: '',
                resume: null
            });
        }
    }, [isOpen]);

    const handleInputChange = (e) => {
        const { name, value } = e.target;
        setFormData(prev => ({
            ...prev,
            [name]: value
        }));
    };

    const handleFileChange = (e) => {
        const file = e.target.files[0];
        if (file) {
            // Check file size (5MB limit)
            if (file.size > 5 * 1024 * 1024) {
                showToast('File size must be less than 5MB', 'error');
                e.target.value = '';
                return;
            }
            
            // Check file type
            const allowedTypes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
            if (!allowedTypes.includes(file.type)) {
                showToast('Please upload a PDF, DOC, or DOCX file', 'error');
                e.target.value = '';
                return;
            }
            
            setFormData(prev => ({
                ...prev,
                resume: file
            }));
        }
    };

    const showToast = (message, type = 'success') => {
        setToast({ show: true, message, type });
        setTimeout(() => {
            setToast({ show: false, message: '', type: 'success' });
        }, 5000);
    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        if (isSubmitting) return;

        setIsSubmitting(true);

        try {
            const submitData = new FormData();
            submitData.append('job_position_id', jobId);
            submitData.append('first_name', formData.first_name);
            submitData.append('last_name', formData.last_name);
            submitData.append('email', formData.email);
            submitData.append('phone', formData.phone);
            submitData.append('cover_letter', formData.cover_letter);
            
            if (formData.resume) {
                submitData.append('resume', formData.resume);
            }
            
            // Add CSRF token
            const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
            submitData.append('_token', csrfToken);
            
            const response = await fetch('/job-applications', {
                method: 'POST',
                body: submitData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });
            
            const data = await response.json();
            
            if (response.ok && data.success) {
                showToast('Application submitted successfully!', 'success');
                setTimeout(() => {
                    onClose();
                }, 2000);
            } else {
                showToast(data.message || 'Failed to submit application', 'error');
            }
            
        } catch (error) {
            console.error('Application submission error:', error);
            showToast('An error occurred while submitting your application', 'error');
        }
        
        setIsSubmitting(false);
    };

    if (!isOpen) return null;

    return React.createElement('div', {
        className: 'fixed inset-0 bg-black bg-opacity-50 modal-overlay flex items-center justify-center p-4',
        style: { 
            transition: 'opacity 0.3s ease-out',
            zIndex: 9999
        },
        onClick: onClose
    }, [
        // Modal Container
        React.createElement('div', {
            key: 'modal',
            className: 'bg-white rounded-3xl shadow-2xl max-w-4xl w-full modal-content',
            style: { 
                maxHeight: '90vh',
                zIndex: 10000
            },
            onClick: (e) => e.stopPropagation()
        }, [
            // Modal Header
            React.createElement('div', {
                key: 'header',
                className: 'flex items-center justify-between p-8 border-b border-gray-200'
            }, [
                React.createElement('div', { key: 'title-section' }, [
                    React.createElement('h2', {
                        key: 'title',
                        className: 'text-2xl font-bold text-aufg-text-primary'
                    }, 'Apply for Position'),
                    React.createElement('p', {
                        key: 'job-title',
                        className: 'text-aufg-text-secondary mt-1'
                    }, jobTitle)
                ]),
                React.createElement('button', {
                    key: 'close-btn',
                    onClick: onClose,
                    className: 'text-gray-400 hover:text-gray-600 text-3xl font-light'
                }, '×')
            ]),

            // Modal Body
            React.createElement('div', {
                key: 'body',
                className: 'overflow-y-auto modal-body-scroll',
                style: { maxHeight: 'calc(90vh - 120px)' }
            }, [
                React.createElement('div', {
                    key: 'body-content',
                    className: 'p-8'
                }, [
                React.createElement('form', {
                    key: 'form',
                    onSubmit: handleSubmit,
                    className: 'space-y-6'
                }, [
                    // Name Fields
                    React.createElement('div', {
                        key: 'name-fields',
                        className: 'grid grid-cols-1 md:grid-cols-2 gap-6'
                    }, [
                        React.createElement('div', { key: 'first-name' }, [
                            React.createElement('label', {
                                key: 'first-name-label',
                                className: 'block text-aufg-text-primary font-semibold mb-2'
                            }, 'First Name *'),
                            React.createElement('input', {
                                key: 'first-name-input',
                                type: 'text',
                                name: 'first_name',
                                value: formData.first_name,
                                onChange: handleInputChange,
                                required: true,
                                className: 'w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-aufg-primary focus:ring-2 focus:ring-aufg-primary/20 outline-none transition-all duration-300',
                                placeholder: 'Enter your first name'
                            })
                        ]),
                        React.createElement('div', { key: 'last-name' }, [
                            React.createElement('label', {
                                key: 'last-name-label',
                                className: 'block text-aufg-text-primary font-semibold mb-2'
                            }, 'Last Name *'),
                            React.createElement('input', {
                                key: 'last-name-input',
                                type: 'text',
                                name: 'last_name',
                                value: formData.last_name,
                                onChange: handleInputChange,
                                required: true,
                                className: 'w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-aufg-primary focus:ring-2 focus:ring-aufg-primary/20 outline-none transition-all duration-300',
                                placeholder: 'Enter your last name'
                            })
                        ])
                    ]),

                    // Contact Fields
                    React.createElement('div', {
                        key: 'contact-fields',
                        className: 'grid grid-cols-1 md:grid-cols-2 gap-6'
                    }, [
                        React.createElement('div', { key: 'email' }, [
                            React.createElement('label', {
                                key: 'email-label',
                                className: 'block text-aufg-text-primary font-semibold mb-2'
                            }, 'Email Address *'),
                            React.createElement('input', {
                                key: 'email-input',
                                type: 'email',
                                name: 'email',
                                value: formData.email,
                                onChange: handleInputChange,
                                required: true,
                                className: 'w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-aufg-primary focus:ring-2 focus:ring-aufg-primary/20 outline-none transition-all duration-300',
                                placeholder: 'your.email@example.com'
                            })
                        ]),
                        React.createElement('div', { key: 'phone' }, [
                            React.createElement('label', {
                                key: 'phone-label',
                                className: 'block text-aufg-text-primary font-semibold mb-2'
                            }, 'Phone Number *'),
                            React.createElement('input', {
                                key: 'phone-input',
                                type: 'tel',
                                name: 'phone',
                                value: formData.phone,
                                onChange: handleInputChange,
                                required: true,
                                className: 'w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-aufg-primary focus:ring-2 focus:ring-aufg-primary/20 outline-none transition-all duration-300',
                                placeholder: '+966 5X XXX XXXX'
                            })
                        ])
                    ]),

                    // Cover Letter
                    React.createElement('div', { key: 'cover-letter' }, [
                        React.createElement('label', {
                            key: 'cover-letter-label',
                            className: 'block text-aufg-text-primary font-semibold mb-2'
                        }, 'Cover Letter'),
                        React.createElement('textarea', {
                            key: 'cover-letter-input',
                            name: 'cover_letter',
                            value: formData.cover_letter,
                            onChange: handleInputChange,
                            rows: 6,
                            className: 'w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-aufg-primary focus:ring-2 focus:ring-aufg-primary/20 outline-none transition-all duration-300 resize-none',
                            placeholder: 'Tell us why you\'re interested in this position and what makes you a great fit...'
                        })
                    ]),

                    // Resume Upload
                    React.createElement('div', { key: 'resume' }, [
                        React.createElement('label', {
                            key: 'resume-label',
                            className: 'block text-aufg-text-primary font-semibold mb-2'
                        }, 'Resume/CV'),
                        React.createElement('input', {
                            key: 'resume-input',
                            type: 'file',
                            onChange: handleFileChange,
                            accept: '.pdf,.doc,.docx',
                            className: 'w-full px-4 py-3 rounded-xl border border-gray-200 focus:border-aufg-primary focus:ring-2 focus:ring-aufg-primary/20 outline-none transition-all duration-300'
                        }),
                        React.createElement('p', {
                            key: 'resume-help',
                            className: 'text-sm text-aufg-text-secondary mt-2'
                        }, 'Accepted formats: PDF, DOC, DOCX (Max 5MB)')
                    ]),

                    // Submit Button
                    React.createElement('div', {
                        key: 'submit-section',
                        className: 'flex justify-end gap-4 pt-6'
                    }, [
                        React.createElement('button', {
                            key: 'cancel-btn',
                            type: 'button',
                            onClick: onClose,
                            className: 'px-6 py-3 border-2 border-gray-300 text-gray-700 rounded-xl font-semibold hover:bg-gray-50 transition-colors'
                        }, 'Cancel'),
                        React.createElement('button', {
                            key: 'submit-btn',
                            type: 'submit',
                            disabled: isSubmitting,
                            className: `bg-aufg-gradient text-white px-8 py-3 rounded-xl font-semibold transition-all duration-300 ${
                                isSubmitting 
                                    ? 'opacity-50 cursor-not-allowed' 
                                    : 'hover:shadow-lg transform hover:scale-105'
                            }`
                        }, isSubmitting ? 'Submitting...' : 'Submit Application')
                    ])
                ])
                ])
            ])
        ]),

        // Toast Notification
        toast.show && React.createElement('div', {
            key: 'toast',
            className: `fixed bottom-4 right-4 ${
                toast.type === 'success' ? 'bg-green-500' : 'bg-red-500'
            } text-white px-6 py-4 rounded-lg shadow-lg`,
            style: { 
                transition: 'all 0.3s ease-out',
                zIndex: 10001
            }
        }, toast.message)
    ]);
}

// Modal Management
let modalState = {
    isOpen: false,
    jobId: null,
    jobTitle: ''
};

function renderModal() {
    const modalRoot = document.getElementById('job-modal-root');
    if (modalRoot) {
        ReactDOM.render(
            React.createElement(JobApplicationModal, {
                isOpen: modalState.isOpen,
                jobId: modalState.jobId,
                jobTitle: modalState.jobTitle,
                onClose: () => {
                    modalState.isOpen = false;
                    renderModal();
                }
            }),
            modalRoot
        );
    }
}

// Global functions to control modal
window.showJobApplicationModal = function(jobId, jobTitle) {
    modalState = {
        isOpen: true,
        jobId: jobId,
        jobTitle: jobTitle
    };
    renderModal();
};

window.hideJobApplicationModal = function() {
    modalState.isOpen = false;
    renderModal();
};

// Initialize modal on page load
document.addEventListener('DOMContentLoaded', function() {
    renderModal();
    console.log('React Job Application Modal initialized');
});

// Character counter for message field
document.addEventListener('DOMContentLoaded', function() {
    const messageField = document.getElementById('message');
    const charCountElement = document.getElementById('charCount');
    
    if (messageField && charCountElement) {
        messageField.addEventListener('input', function() {
            const remaining = 255 - this.value.length;
            charCountElement.textContent = remaining + ' remaining';
            
            // Change color based on remaining characters
            if (remaining < 50) {
                charCountElement.style.color = '#dc2626';
            } else if (remaining < 100) {
                charCountElement.style.color = '#f59e0b';
            } else {
                charCountElement.style.color = '#038BB9';
            }
        });
    }

    // File upload feedback
    const fileInput = document.getElementById('attachment');
    const fileLabel = document.querySelector('label[for="attachment"]');
    
    if (fileInput && fileLabel) {
        fileInput.addEventListener('change', function() {
            if (this.files.length > 0) {
                const fileName = this.files[0].name;
                fileLabel.querySelector('p').textContent = `Selected: ${fileName}`;
            }
        });
    }

    // Hiring Process Animation Script
    console.log('Setting up hiring process animations...');
    
    const hiringSection = document.getElementById('hiring-process');
    if (!hiringSection) {
        console.log('Hiring process section not found');
        return;
    }
    
    // Add classes to elements for animation  
    const processSteps = hiringSection.querySelectorAll('.space-y-16 > div');
    const centralLine = hiringSection.querySelector('.absolute.left-1\\/2.transform.-translate-x-1\\/2');
    
    // Add animation classes to steps
    processSteps.forEach((step, index) => {
        step.classList.add('hiring-process-step');
        
        // Add classes to cards and icons for hover effects
        const card = step.querySelector('.bg-white.rounded-3xl');
        const numberBadge = step.querySelector('.w-12.h-12.bg-aufg-gradient');
        
        if (card) card.classList.add('hiring-step-card');
        if (numberBadge) numberBadge.classList.add('hiring-step-number');
    });
    
    // Add class to central line
    if (centralLine) {
        centralLine.classList.add('hiring-process-line');
    }
    
    // Create intersection observer for animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -100px 0px'
    };
    
    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                console.log('Hiring process section is visible, starting animations');
                
                // Animate central line first
                if (centralLine) {
                    setTimeout(() => {
                        centralLine.classList.add('animate-in');
                    }, 200);
                }
                
                // Animate steps with staggered timing
                processSteps.forEach((step, index) => {
                    setTimeout(() => {
                        step.classList.add('animate-in');
                        console.log(`Step ${index + 1} animated in`);
                        
                        // Add active state to first step briefly
                        if (index === 0) {
                            setTimeout(() => {
                                step.classList.add('hiring-step-active');
                                setTimeout(() => {
                                    step.classList.remove('hiring-step-active');
                                }, 3000);
                            }, 500);
                        }
                    }, 300 + (index * 200));
                });
                
                // Stop observing after animation
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);
    
    // Start observing the hiring section
    if (hiringSection) {
        observer.observe(hiringSection);
    }
    
    // Add click interaction for steps
    processSteps.forEach((step, index) => {
        step.addEventListener('click', function() {
            // Remove active class from all steps
            processSteps.forEach(s => s.classList.remove('hiring-step-active'));
            // Add active class to clicked step
            this.classList.add('hiring-step-active');
            console.log(`Step ${index + 1} activated`);
        });
    });
    
    console.log('Hiring process animations initialized successfully');

    // General scroll animations
    const generalObserverOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const generalObserver = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-fade-in-up');
            }
        });
    }, generalObserverOptions);
    
    // Observe all sections
    document.querySelectorAll('section').forEach(section => {
        generalObserver.observe(section);
    });
});