// Mobile Product Tabs Handler
document.addEventListener('DOMContentLoaded', function() {
    // Only run on mobile devices
    function isMobile() {
        return window.innerWidth <= 768;
    }
    
    if (!isMobile()) {
        console.log('Desktop detected - skipping mobile tab handler');
        return;
    }
    
    console.log('Mobile detected - initializing simple product tabs');
    
    // Find tab elements
    const allTab = document.querySelector('[data-id="all-tab"]');
    const floatGlassTab = document.querySelector('[data-id="4cf582a"]'); 
    const patternGlassTab = document.querySelector('[data-id="66e040e"]');
    const mirrorTab = document.querySelector('[data-id="be30b34"]');
    
    // Find product elements
    const clearFloatContainer = document.querySelector('[data-id="5f877d7"]');
    const silverMirrorContainer = document.querySelector('[data-id="95bfa7a"]');
    const patternGlassContainer = document.querySelector('[data-id="5a8a143"]');
    const decorativeMirrorContainer = document.querySelector('[data-id="dec8f47"]');
    
    const allTabs = [allTab, floatGlassTab, patternGlassTab, mirrorTab].filter(tab => tab !== null);
    const allProducts = [clearFloatContainer, silverMirrorContainer, patternGlassContainer, decorativeMirrorContainer].filter(p => p !== null);
    
    if (allTabs.length === 0 || allProducts.length === 0) {
        console.log('Tabs or products not found - skipping mobile tab setup');
        return;
    }
    
    // Set active tab styling
    function setActiveTab(activeTab) {
        allTabs.forEach(tab => {
            tab.classList.remove('active-product-tab', 'active-tab-wrapper');
            const heading = tab.querySelector('.tab-heading');
            if (heading) {
                heading.style.color = '#038BB9';
            }
        });
        
        if (activeTab) {
            activeTab.classList.add('active-product-tab', 'active-tab-wrapper');
            const heading = activeTab.querySelector('.tab-heading');
            if (heading) {
                heading.style.color = 'white';
            }
        }
    }
    
    // Enhanced mobile product filtering with smooth animations
    function filterProducts(category) {
        console.log(`Mobile filtering for category: ${category}`);
        
        let visibleProducts = [];
        
        switch(category) {
            case 'all':
                visibleProducts = allProducts;
                break;
            case 'float':
                if (clearFloatContainer) visibleProducts = [clearFloatContainer];
                break;
            case 'pattern':
                if (patternGlassContainer) visibleProducts = [patternGlassContainer];
                break;
            case 'mirror':
                visibleProducts = [silverMirrorContainer, decorativeMirrorContainer].filter(p => p !== null);
                break;
        }
        
        const productsToHide = allProducts.filter(p => !visibleProducts.includes(p));
        
        // Smooth fade out animation
        productsToHide.forEach((product, index) => {
            setTimeout(() => {
                product.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
                product.style.opacity = '0';
                product.style.transform = 'translateY(-10px)';
                
                setTimeout(() => {
                    product.style.display = 'none';
                }, 300);
            }, index * 50);
        });
        
        // Smooth fade in animation for visible products
        setTimeout(() => {
            visibleProducts.forEach((product, index) => {
                product.style.display = 'flex';
                product.style.opacity = '0';
                product.style.transform = 'translateY(10px)';
                
                setTimeout(() => {
                    product.style.transition = 'opacity 0.4s ease, transform 0.4s ease';
                    product.style.opacity = '1';
                    product.style.transform = 'translateY(0)';
                }, index * 100);
            });
        }, 200);
        
        console.log(`Mobile filter: Showing ${visibleProducts.length} products`);
    }
    
    // Add tab click handlers
    if (allTab) {
        allTab.addEventListener('click', function(e) {
            e.preventDefault();
            setActiveTab(this);
            filterProducts('all');
        });
    }
    
    if (floatGlassTab) {
        floatGlassTab.addEventListener('click', function(e) {
            e.preventDefault();
            setActiveTab(this);
            filterProducts('float');
        });
    }
    
    if (patternGlassTab) {
        patternGlassTab.addEventListener('click', function(e) {
            e.preventDefault();
            setActiveTab(this);
            filterProducts('pattern');
        });
    }
    
    if (mirrorTab) {
        mirrorTab.addEventListener('click', function(e) {
            e.preventDefault();
            setActiveTab(this);
            filterProducts('mirror');
        });
    }
    
    // Add touch feedback for better mobile experience
    function addTouchFeedback(element) {
        element.addEventListener('touchstart', function() {
            this.style.transform = 'scale(0.95)';
        }, { passive: true });
        
        element.addEventListener('touchend', function() {
            this.style.transform = 'scale(1)';
        }, { passive: true });
        
        element.addEventListener('touchcancel', function() {
            this.style.transform = 'scale(1)';
        }, { passive: true });
    }
    
    // Apply touch feedback to all tabs
    allTabs.forEach(tab => {
        if (tab) {
            addTouchFeedback(tab);
        }
    });
    
    // Add touch feedback to product cards
    allProducts.forEach(product => {
        if (product) {
            addTouchFeedback(product);
        }
    });
    
    // Initialize - show all products and set all tab as active
    filterProducts('all');
    if (allTab) {
        setActiveTab(allTab);
    }
    
    // Handle window resize
    window.addEventListener('resize', function() {
        if (!isMobile()) {
            // Switching to desktop - clean up and show all products
            allProducts.forEach(product => {
                product.style.display = '';
                product.style.opacity = '';
            });
            console.log('Switched to desktop - mobile tab handler disabled');
        }
    });
    
    console.log('Mobile product tabs initialized successfully');
});