@extends('layouts.admin-simple')

@section('title', 'Edit News Article')
@section('page-title', 'Edit News Article')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Dashboard</a></li>
    <li class="breadcrumb-item"><a href="{{ route('admin.news') }}">News</a></li>
    <li class="breadcrumb-item active" aria-current="page">Edit Article</li>
@endsection

@push('styles')
<link href="https://cdn.jsdelivr.net/npm/summernote@0.8.18/dist/summernote-bs4.min.css" rel="stylesheet">
@endpush

@section('content')
    <form method="POST" action="{{ route('admin.news.update', $news) }}" enctype="multipart/form-data" id="editNewsForm">
        @csrf
        @method('PUT')
        
        <div class="row">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header bg-white">
                        <h4 class="card-header__title">Article Content</h4>
                    </div>
                    <div class="card-body">
                        <div class="form-group">
                            <label for="title">Title <span class="text-danger">*</span></label>
                            <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                   id="title" name="title" value="{{ old('title', $news->title) }}" required>
                            @error('title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="form-group">
                            <label for="excerpt">Excerpt</label>
                            <textarea class="form-control @error('excerpt') is-invalid @enderror" 
                                      id="excerpt" name="excerpt" rows="3" 
                                      placeholder="Brief summary of the article (optional)">{{ old('excerpt', $news->excerpt) }}</textarea>
                            <small class="form-text text-muted">A short description that appears in news listings</small>
                            @error('excerpt')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="form-group">
                            <label for="content">Content <span class="text-danger">*</span></label>
                            <textarea class="form-control @error('content') is-invalid @enderror" 
                                      id="content" name="content" rows="15" required>{{ old('content', $news->content) }}</textarea>
                            @error('content')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="form-group">
                            <label for="image">Featured Image</label>
                            @if($news->image)
                                <div class="mb-2">
                                    <img src="{{ $news->featured_image_url }}" alt="Current image" 
                                         class="img-thumbnail" style="max-width: 200px;">
                                    <p class="small text-muted">Current image</p>
                                </div>
                            @endif
                            <input type="file" class="form-control-file @error('image') is-invalid @enderror" 
                                   id="image" name="image" accept="image/*">
                            <small class="form-text text-muted">Recommended size: 1200x630px (JPG, PNG, max 2MB). Leave empty to keep current image.</small>
                            @error('image')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
                
                <!-- SEO Settings -->
                <div class="card mt-3">
                    <div class="card-header bg-white">
                        <h5 class="card-header__title">SEO Settings</h5>
                    </div>
                    <div class="card-body">
                        <div class="form-group">
                            <label for="meta_title">Meta Title</label>
                            <input type="text" class="form-control @error('meta_title') is-invalid @enderror" 
                                   id="meta_title" name="meta_title" value="{{ old('meta_title', $news->meta_title) }}" 
                                   placeholder="SEO optimized title">
                            @error('meta_title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="form-group">
                            <label for="meta_description">Meta Description</label>
                            <textarea class="form-control @error('meta_description') is-invalid @enderror" 
                                      id="meta_description" name="meta_description" rows="3" 
                                      placeholder="SEO description for search engines">{{ old('meta_description', $news->meta_description) }}</textarea>
                            @error('meta_description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header bg-white">
                        <h5 class="card-header__title">Publishing Options</h5>
                    </div>
                    <div class="card-body">
                        <div class="form-group">
                            <label for="category_id">Category <span class="text-danger">*</span></label>
                            <select class="form-control @error('category_id') is-invalid @enderror" 
                                    id="category_id" name="category_id" required>
                                <option value="">Select Category</option>
                                @foreach($categories as $category)
                                    <option value="{{ $category->id }}" {{ old('category_id', $news->category_id) == $category->id ? 'selected' : '' }}>
                                        {{ $category->name }}
                                    </option>
                                @endforeach
                            </select>
                            @error('category_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="form-group">
                            <label for="tags">Tags</label>
                            <input type="text" class="form-control @error('tags') is-invalid @enderror" 
                                   id="tags" name="tags" value="{{ old('tags', $news->tags ? implode(', ', $news->tags) : '') }}" 
                                   placeholder="tag1, tag2, tag3">
                            <small class="form-text text-muted">Separate tags with commas</small>
                            @error('tags')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="form-group">
                            <label for="published_at">Publish Date</label>
                            <input type="datetime-local" class="form-control @error('published_at') is-invalid @enderror" 
                                   id="published_at" name="published_at" 
                                   value="{{ old('published_at', $news->published_at ? $news->published_at->format('Y-m-d\TH:i') : '') }}">
                            <small class="form-text text-muted">Leave empty to publish immediately</small>
                            @error('published_at')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        
                        <div class="form-group">
                            <div class="custom-control custom-checkbox mb-2">
                                <input type="checkbox" class="custom-control-input" id="is_published" 
                                       name="is_published" value="1" {{ old('is_published', $news->is_published) ? 'checked' : '' }}>
                                <label class="custom-control-label" for="is_published">Published</label>
                            </div>
                            
                            <div class="custom-control custom-checkbox">
                                <input type="checkbox" class="custom-control-input" id="is_featured" 
                                       name="is_featured" value="1" {{ old('is_featured', $news->is_featured) ? 'checked' : '' }}>
                                <label class="custom-control-label" for="is_featured">Featured</label>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card mt-3">
                    <div class="card-header bg-white">
                        <h5 class="card-header__title">Actions</h5>
                    </div>
                    <div class="card-body">
                        <button type="submit" class="btn btn-primary btn-block">
                            <i class="material-icons mr-2">save</i> Update Article
                        </button>
                        <a href="{{ route('admin.news') }}" class="btn btn-light btn-block">
                            <i class="material-icons mr-2">arrow_back</i> Back to News
                        </a>
                        <a href="{{ route('news.show', $news->slug) }}" class="btn btn-outline-info btn-block" target="_blank">
                            <i class="material-icons mr-2">visibility</i> Preview Article
                        </a>
                    </div>
                </div>
                
                <div class="card mt-3">
                    <div class="card-header bg-white">
                        <h5 class="card-header__title">Article Info</h5>
                    </div>
                    <div class="card-body">
                        <dl class="row">
                            <dt class="col-sm-5">Created:</dt>
                            <dd class="col-sm-7">{{ $news->created_at->format('M d, Y H:i') }}</dd>
                            
                            <dt class="col-sm-5">Last Updated:</dt>
                            <dd class="col-sm-7">{{ $news->updated_at->format('M d, Y H:i') }}</dd>
                            
                            <dt class="col-sm-5">Author:</dt>
                            <dd class="col-sm-7">{{ $news->author->full_name ?? 'Unknown' }}</dd>
                            
                            <dt class="col-sm-5">Slug:</dt>
                            <dd class="col-sm-7"><code>{{ $news->slug }}</code></dd>
                            
                            @if($news->published_at)
                            <dt class="col-sm-5">Published:</dt>
                            <dd class="col-sm-7">{{ $news->published_at->format('M d, Y H:i') }}</dd>
                            @endif
                        </dl>
                    </div>
                </div>
            </div>
        </div>
    </form>
@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/summernote@0.8.18/dist/summernote-bs4.min.js"></script>
<script>
$(document).ready(function() {
    // Initialize Summernote
    $('#content').summernote({
        height: 400,
        toolbar: [
            ['style', ['style']],
            ['font', ['bold', 'underline', 'clear']],
            ['color', ['color']],
            ['para', ['ul', 'ol', 'paragraph']],
            ['table', ['table']],
            ['insert', ['link', 'picture', 'video']],
            ['view', ['fullscreen', 'codeview', 'help']]
        ]
    });
});
</script>
@endpush