@extends('layouts.admin-simple')

@section('title', 'Inventory Aging Report')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item">Inventory</li>
    <li class="breadcrumb-item active">Aging Report</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Inventory Aging Report</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Analyze inventory based on age and identify slow-moving items</p>
        </div>
        @can('warehouse.inventory.view')
        <div class="page-actions-right d-flex align-items-center">
            <button type="button" onclick="refreshAgingReport()" class="btn btn-sm btn-success mr-2" title="Refresh aging data">
                <i class="material-icons mr-1" style="font-size: 16px;">refresh</i>
                Refresh
            </button>

            <button type="button" onclick="exportAgingReport()" class="btn btn-sm btn-info mr-2" title="Export aging report to Excel">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>

            <button type="button" onclick="printAgingReport()" class="btn btn-sm btn-primary" title="Print aging report">
                <i class="material-icons mr-1" style="font-size: 16px;">print</i>
                Print
            </button>
        </div>
        @endcan
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Fresh Items (0-30 Days)"
                :value="collect($agingInventory)->where('age_days', '<=', 30)->count()"
                icon="new_releases"
                color="success"
                subtitle="Recent inventory"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Moderate Age (31-90 Days)"
                :value="collect($agingInventory)->whereBetween('age_days', [31, 90])->count()"
                icon="schedule"
                color="info"
                subtitle="Moderately aged items"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Slow Moving (91-180 Days)"
                :value="collect($agingInventory)->whereBetween('age_days', [91, 180])->count()"
                icon="slow_motion_video"
                color="warning"
                subtitle="Requires attention"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Dead Stock (180+ Days)"
                :value="collect($agingInventory)->where('age_days', '>', 180)->count()"
                icon="block"
                color="danger"
                subtitle="Immediate action needed"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Aging Report Filters
            </h5>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('warehouse.inventory.aging-report') }}">
                <div class="row">
                    <div class="col-xl-3 col-md-6 mb-3">
                        <label class="form-label">Aging Period</label>
                        <select name="days" id="days" class="form-control">
                            <option value="30" {{ $days == 30 ? 'selected' : '' }}>30 Days</option>
                            <option value="60" {{ $days == 60 ? 'selected' : '' }}>60 Days</option>
                            <option value="90" {{ $days == 90 ? 'selected' : '' }}>90 Days</option>
                            <option value="180" {{ $days == 180 ? 'selected' : '' }}>180 Days</option>
                            <option value="365" {{ $days == 365 ? 'selected' : '' }}>1 Year</option>
                        </select>
                    </div>

                    <div class="col-xl-3 col-md-6 mb-3">
                        <label class="form-label">Category</label>
                        <select id="filter-category" class="form-control">
                            <option value="">All Categories</option>
                            @foreach(collect($agingInventory)->pluck('category')->unique()->filter() as $category)
                                <option value="{{ $category }}">{{ $category }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-xl-3 col-md-6 mb-3">
                        <label class="form-label">Age Group</label>
                        <select id="filter-age-group" class="form-control">
                            <option value="">All Age Groups</option>
                            <option value="fresh">Fresh (0-30 days)</option>
                            <option value="moderate">Moderate (31-90 days)</option>
                            <option value="slow">Slow Moving (91-180 days)</option>
                            <option value="dead">Dead Stock (180+ days)</option>
                        </select>
                    </div>

                    <div class="col-xl-3 col-md-6 mb-3">
                        <label class="form-label">Item Search</label>
                        <input type="text" id="filter-search" class="form-control" placeholder="Search by item name or code">
                    </div>
                </div>

                <div class="d-flex justify-content-between align-items-center mt-3 pt-3 border-top">
                    <button type="submit" class="btn btn-primary">
                        <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">update</i>
                        Update Report
                    </button>

                    <button type="button" onclick="clearAgingFilters()" class="btn btn-outline-secondary">
                        <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">clear</i>
                        Clear Filters
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Age Distribution Chart Card -->
    <div class="row mb-4">
        <div class="col-lg-12">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0 d-flex align-items-center">
                        <i class="material-icons mr-2">pie_chart</i>
                        Age Distribution Analysis
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-lg-3 col-md-6 mb-3">
                            <div class="text-center">
                                <div class="h4 mb-1 text-success">{{ collect($agingInventory)->where('age_days', '<=', 30)->count() }}</div>
                                <div class="small text-muted">Fresh Items</div>
                                <div class="small text-success">
                                    Value: د.إ {{ number_format(collect($agingInventory)->where('age_days', '<=', 30)->sum('value'), 2) }}
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-md-6 mb-3">
                            <div class="text-center">
                                <div class="h4 mb-1 text-info">{{ collect($agingInventory)->whereBetween('age_days', [31, 90])->count() }}</div>
                                <div class="small text-muted">Moderate Age</div>
                                <div class="small text-info">
                                    Value: د.إ {{ number_format(collect($agingInventory)->whereBetween('age_days', [31, 90])->sum('value'), 2) }}
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-md-6 mb-3">
                            <div class="text-center">
                                <div class="h4 mb-1 text-warning">{{ collect($agingInventory)->whereBetween('age_days', [91, 180])->count() }}</div>
                                <div class="small text-muted">Slow Moving</div>
                                <div class="small text-warning">
                                    Value: د.إ {{ number_format(collect($agingInventory)->whereBetween('age_days', [91, 180])->sum('value'), 2) }}
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-md-6 mb-3">
                            <div class="text-center">
                                <div class="h4 mb-1 text-danger">{{ collect($agingInventory)->where('age_days', '>', 180)->count() }}</div>
                                <div class="small text-muted">Dead Stock</div>
                                <div class="small text-danger">
                                    Value: د.إ {{ number_format(collect($agingInventory)->where('age_days', '>', 180)->sum('value'), 2) }}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Aging Report Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">list</i>
                Aging Report Details ({{ collect($agingInventory)->count() }} items)
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            @if(collect($agingInventory)->count() > 0)
                <div class="table-responsive">
                    <table id="aging-table" class="table table-hover">
                        <thead>
                            <tr>
                                <th>Item</th>
                                <th>Category</th>
                                <th>Current Stock</th>
                                <th>Age (Days)</th>
                                <th>Last Transaction</th>
                                <th>Value (د.إ)</th>
                                <th>Age Category</th>
                                <th class="text-right">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach(collect($agingInventory) as $item)
                                @php
                                    $agingCategory = '';
                                    $badgeClass = '';
                                    $rowClass = '';
                                    if ($item['age_days'] <= 30) {
                                        $agingCategory = 'Fresh';
                                        $badgeClass = 'badge-success';
                                        $rowClass = 'table-success-light';
                                    } elseif ($item['age_days'] <= 90) {
                                        $agingCategory = 'Moderate';
                                        $badgeClass = 'badge-info';
                                        $rowClass = 'table-info-light';
                                    } elseif ($item['age_days'] <= 180) {
                                        $agingCategory = 'Slow Moving';
                                        $badgeClass = 'badge-warning';
                                        $rowClass = 'table-warning-light';
                                    } else {
                                        $agingCategory = 'Dead Stock';
                                        $badgeClass = 'badge-danger';
                                        $rowClass = 'table-danger-light';
                                    }
                                @endphp
                                <tr class="aging-row {{ $rowClass }}" data-age-days="{{ $item['age_days'] }}" data-category="{{ $item['category'] ?? '' }}">
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="mr-3">
                                                <div class="bg-primary text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                                    <span class="font-weight-bold" style="font-size: 14px;">{{ substr($item['item_name'] ?? 'U', 0, 1) }}</span>
                                                </div>
                                            </div>
                                            <div>
                                                <div class="font-weight-medium">{{ $item['item_name'] ?? 'Unknown Item' }}</div>
                                                <small class="text-muted">{{ $item['item_code'] ?? 'No Code' }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        @if($item['category'] ?? null)
                                            <div>
                                                <i class="material-icons text-muted mr-1" style="font-size: 14px; vertical-align: middle;">category</i>
                                                <span>{{ $item['category'] }}</span>
                                            </div>
                                        @else
                                            <span class="text-muted">-</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div>
                                            <div class="font-weight-medium mb-1">{{ number_format($item['current_stock'] ?? 0, 2) }}</div>
                                            <small class="text-muted">units</small>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="badge {{ $badgeClass }} age-badge" data-age="{{ $item['age_days'] ?? 0 }}">
                                            {{ $item['age_days'] ?? 0 }} days
                                        </span>
                                    </td>
                                    <td>
                                        <div>
                                            <div class="small">{{ $item['last_transaction'] ?? 'N/A' }}</div>
                                        </div>
                                    </td>
                                    <td class="text-right">
                                        <div class="font-weight-medium">{{ number_format($item['value'] ?? 0, 2) }}</div>
                                    </td>
                                    <td>
                                        <span class="badge {{ $badgeClass }}">{{ $agingCategory }}</span>
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <button type="button"
                                                    class="btn btn-sm btn-outline-primary view-item-btn"
                                                    title="View Item Details"
                                                    data-item-code="{{ $item['item_code'] ?? '' }}">
                                                <i class="material-icons" style="font-size: 16px;">visibility</i>
                                            </button>
                                            @can('warehouse.inventory.adjust')
                                            <button type="button"
                                                    class="btn btn-sm btn-outline-warning adjust-stock-btn"
                                                    title="Adjust Stock"
                                                    data-item-code="{{ $item['item_code'] ?? '' }}">
                                                <i class="material-icons" style="font-size: 16px;">tune</i>
                                            </button>
                                            @endcan
                                            <button type="button"
                                                    class="btn btn-sm btn-outline-info history-btn"
                                                    title="View History"
                                                    data-item-code="{{ $item['item_code'] ?? '' }}">
                                                <i class="material-icons" style="font-size: 16px;">history</i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                        <tfoot>
                            <tr class="table-info">
                                <th colspan="5" class="text-right">Total Items:</th>
                                <th class="text-right">
                                    <strong>د.إ {{ number_format(collect($agingInventory)->sum('value'), 2) }}</strong>
                                </th>
                                <th colspan="2">
                                    <strong>{{ collect($agingInventory)->count() }} items</strong>
                                </th>
                            </tr>
                        </tfoot>
                    </table>
                </div>

                <!-- Summary and Recommendations -->
                <div class="row mt-4">
                    <div class="col-md-6">
                        <div class="card border-left-info">
                            <div class="card-body">
                                <h6 class="font-weight-bold text-info">Report Summary</h6>
                                <ul class="mb-0">
                                    <li>Total Items Analyzed: {{ collect($agingInventory)->count() }}</li>
                                    <li>Average Age: {{ collect($agingInventory)->avg('age_days') ? round(collect($agingInventory)->avg('age_days')) : 0 }} days</li>
                                    <li>Total Value: د.إ {{ number_format(collect($agingInventory)->sum('value'), 2) }}</li>
                                    <li>Report Generated: {{ now()->format('F j, Y g:i A') }}</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card border-left-warning">
                            <div class="card-body">
                                <h6 class="font-weight-bold text-warning">Recommendations</h6>
                                <ul class="mb-0">
                                    @if(collect($agingInventory)->where('age_days', '>', 180)->count() > 0)
                                        <li>Review {{ collect($agingInventory)->where('age_days', '>', 180)->count() }} dead stock items for disposal</li>
                                    @endif
                                    @if(collect($agingInventory)->whereBetween('age_days', [91, 180])->count() > 0)
                                        <li>Monitor {{ collect($agingInventory)->whereBetween('age_days', [91, 180])->count() }} slow-moving items</li>
                                    @endif
                                    <li>Consider promotional sales for aged inventory</li>
                                    <li>Review reorder points for slow-moving items</li>
                                    <li>Implement FIFO inventory rotation practices</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            @else
                <div class="text-center py-5">
                    <i class="material-icons text-success mb-3" style="font-size: 48px;">check_circle</i>
                    <h5 class="text-muted">No Aging Inventory Found</h5>
                    <p class="text-muted">All inventory items are within the acceptable age range of {{ $days }} days.</p>
                    <a href="{{ route('warehouse.inventory.index') }}" class="btn btn-primary">
                        <i class="material-icons mr-1">inventory</i>
                        View Current Inventory
                    </a>
                </div>
            @endif
        </div>
    </div>
@endsection

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <style>
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
            border-radius: 4px;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .aging-row:hover {
            background-color: #f8f9fa;
        }

        .table-success-light {
            background-color: rgba(40, 167, 69, 0.1);
        }

        .table-info-light {
            background-color: rgba(23, 162, 184, 0.1);
        }

        .table-warning-light {
            background-color: rgba(255, 193, 7, 0.1);
        }

        .table-danger-light {
            background-color: rgba(220, 53, 69, 0.1);
        }

        .age-badge {
            font-weight: 500;
            font-size: 0.75rem;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        @media print {
            .btn, .card-header, .no-print, .page-actions-right {
                display: none !important;
            }
            .card {
                border: none !important;
                box-shadow: none !important;
            }
            .table-responsive {
                overflow: visible !important;
            }
        }

        .distribution-card {
            border-left: 4px solid #007bff;
            transition: all 0.15s ease-in-out;
        }

        .distribution-card:hover {
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>

    <script>
        let agingTable;

        $(document).ready(function() {
            initializeDataTable();
            initializeEventHandlers();

            // Entries per page change handler
            $('#entries-per-page').change(function() {
                agingTable.page.len($(this).val()).draw();
            });

            // Auto-refresh on aging period change
            $('#days').change(function() {
                $(this).closest('form').submit();
            });
        });

        function initializeDataTable() {
            agingTable = $('#aging-table').DataTable({
                processing: false,
                serverSide: false,
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                order: [[3, 'desc']], // Sort by age descending
                language: {
                    processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
                    emptyTable: 'No aging inventory found',
                    zeroRecords: 'No matching items found'
                },
                columnDefs: [
                    { orderable: false, targets: [7] }, // Actions column
                    { searchable: false, targets: [7] }, // Actions column
                    { type: 'num', targets: [2, 3, 5] }, // Numeric columns
                    { className: 'text-right', targets: [5] } // Right-align value column
                ],
                drawCallback: function(settings) {
                    $('#entries-per-page').val(settings._iDisplayLength);
                    initializeEventHandlers(); // Re-initialize handlers for new DOM elements
                }
            });
        }

        function initializeEventHandlers() {
            // View item button
            $('.view-item-btn').off('click').on('click', function() {
                const itemCode = $(this).data('item-code');
                showItemDetails(itemCode);
            });

            // Adjust stock button
            $('.adjust-stock-btn').off('click').on('click', function() {
                const itemCode = $(this).data('item-code');
                adjustItemStock(itemCode);
            });

            // History button
            $('.history-btn').off('click').on('click', function() {
                const itemCode = $(this).data('item-code');
                showItemHistory(itemCode);
            });
        }

        function applyAgingFilters() {
            const category = $('#filter-category').val();
            const ageGroup = $('#filter-age-group').val();
            const search = $('#filter-search').val();

            // Apply DataTable search
            agingTable.search(search);

            // Apply custom filters
            $.fn.dataTable.ext.search.push(function(settings, data, dataIndex) {
                if (settings.nTable.id !== 'aging-table') {
                    return true;
                }

                const row = agingTable.row(dataIndex).node();
                const rowCategory = $(row).data('category') || '';
                const rowAgeDays = parseInt($(row).data('age-days')) || 0;

                // Category filter
                if (category && rowCategory.indexOf(category) === -1) {
                    return false;
                }

                // Age group filter
                if (ageGroup) {
                    switch (ageGroup) {
                        case 'fresh':
                            if (rowAgeDays > 30) return false;
                            break;
                        case 'moderate':
                            if (rowAgeDays <= 30 || rowAgeDays > 90) return false;
                            break;
                        case 'slow':
                            if (rowAgeDays <= 90 || rowAgeDays > 180) return false;
                            break;
                        case 'dead':
                            if (rowAgeDays <= 180) return false;
                            break;
                    }
                }

                return true;
            });

            agingTable.draw();
        }

        function clearAgingFilters() {
            $('#filter-category, #filter-age-group, #filter-search').val('');

            // Clear custom filters
            $.fn.dataTable.ext.search.pop();

            agingTable.search('').draw();
        }

        function refreshAgingReport() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Refreshing...';
            btn.disabled = true;

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                location.reload();
            }, 1500);
        }

        function exportAgingReport() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Exporting...';
            btn.disabled = true;

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                showAlert('Export functionality will be implemented', 'info');
            }, 800);
        }

        function printAgingReport() {
            window.print();
        }

        function showItemDetails(itemCode) {
            showAlert(`Item details for ${itemCode} will be implemented`, 'info');
        }

        function adjustItemStock(itemCode) {
            showAlert(`Stock adjustment for ${itemCode} will be implemented`, 'info');
        }

        function showItemHistory(itemCode) {
            showAlert(`Item history for ${itemCode} will be implemented`, 'info');
        }

        function showAlert(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : type === 'info' ? 'alert-info' : 'alert-danger';
            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            `;
            $('body').append(alert);
            setTimeout(() => $('.alert').alert('close'), 5000);
        }

        // Apply filters when filter controls change
        $('#filter-category, #filter-age-group, #filter-search').on('change keyup', function() {
            applyAgingFilters();
        });
    </script>
@endpush