@extends('layouts.admin-simple')

@section('title', 'Items')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item active">Items</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Items</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Manage inventory items and their details</p>
        </div>
        @can('warehouse.items.create')
        <div class="page-actions-right d-flex align-items-center">
            <button onclick="showAddItemModal()" class="btn btn-sm btn-primary" title="Add new item">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                Add Item
            </button>
        </div>
        @endcan
    </div>
@endsection

@section('content')
    <!-- Filters Row -->
    <div class="card mb-3">
        <div class="card-body py-2">
            <div class="row align-items-center">
                <div class="col-md-3">
                    <label class="text-muted mb-1" style="font-size: 12px;">Category</label>
                    <select id="filter-category" class="form-control form-control-sm">
                        <option value="">All Categories</option>
                        @php
                            $categories = \App\Models\Warehouse\ItemCategory::where('status', 'active')->get();
                        @endphp
                        @foreach($categories as $category)
                            <option value="{{ $category->id }}">{{ $category->category_name }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="text-muted mb-1" style="font-size: 12px;">Status</label>
                    <select id="filter-status" class="form-control form-control-sm">
                        <option value="">All Statuses</option>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                        <option value="discontinued">Discontinued</option>
                    </select>
                </div>
                <div class="col-md-6 text-right">
                    <button onclick="applyFilters()" class="btn btn-sm btn-secondary mr-2">
                        <i class="material-icons mr-1" style="font-size: 14px;">filter_list</i>Apply Filters
                    </button>
                    <button onclick="clearFilters()" class="btn btn-sm btn-outline-secondary">
                        <i class="material-icons mr-1" style="font-size: 14px;">clear</i>Clear
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Items Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">inventory_2</i>
                Items List
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="items-table" class="table table-hover">
                    <thead>
                        <tr>
                            <th>Item Info</th>
                            <th>Category</th>
                            <th>Unit</th>
                            <th>Stock Levels</th>
                            <th>Status</th>
                            <th>Created Date</th>
                            <th class="text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Data will be populated by DataTables -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Add Item Modal -->
    <div class="modal fade" id="addItemModal" tabindex="-1" role="dialog" aria-labelledby="addItemModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addItemModalLabel">
                        <i class="material-icons mr-2">inventory_2</i>
                        Add New Item
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <form id="addItemForm">
                        @csrf
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="item_code">Item Code</label>
                                    <div class="input-group">
                                        <input type="text" name="item_code" id="item_code" class="form-control" placeholder="Auto-generated if empty">
                                        <div class="input-group-append">
                                            <button type="button" onclick="generateItemCode()" class="btn btn-outline-secondary" title="Generate Code">
                                                <i class="material-icons" style="font-size: 16px;">refresh</i>
                                            </button>
                                        </div>
                                    </div>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="category_id">Category <span class="text-danger">*</span></label>
                                    <select name="category_id" id="category_id" class="form-control" required>
                                        <option value="">Select Category</option>
                                        @foreach($categories as $category)
                                            <option value="{{ $category->id }}">{{ $category->category_name }}</option>
                                        @endforeach
                                    </select>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <div class="form-group">
                                    <label for="item_description">Item Description <span class="text-danger">*</span></label>
                                    <input type="text" name="item_description" id="item_description" class="form-control" required>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="unit_of_measure">Unit of Measure <span class="text-danger">*</span></label>
                                    <select name="unit_of_measure" id="unit_of_measure" class="form-control" required>
                                        <option value="">Select Unit</option>
                                        <option value="pcs">Pieces (pcs)</option>
                                        <option value="kg">Kilograms (kg)</option>
                                        <option value="g">Grams (g)</option>
                                        <option value="l">Liters (l)</option>
                                        <option value="ml">Milliliters (ml)</option>
                                        <option value="m">Meters (m)</option>
                                        <option value="cm">Centimeters (cm)</option>
                                        <option value="box">Box</option>
                                        <option value="pack">Pack</option>
                                        <option value="set">Set</option>
                                    </select>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="status">Status <span class="text-danger">*</span></label>
                                    <select name="status" id="status" class="form-control" required>
                                        <option value="active">Active</option>
                                        <option value="inactive">Inactive</option>
                                        <option value="discontinued">Discontinued</option>
                                    </select>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="min_stock_level">Min Stock Level</label>
                                    <input type="number" name="min_stock_level" id="min_stock_level" class="form-control" min="0" step="0.01">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="max_stock_level">Max Stock Level</label>
                                    <input type="number" name="max_stock_level" id="max_stock_level" class="form-control" min="0" step="0.01">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="reorder_point">Reorder Point</label>
                                    <input type="number" name="reorder_point" id="reorder_point" class="form-control" min="0" step="0.01">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <div class="form-group">
                                    <label for="notes">Notes</label>
                                    <textarea name="notes" id="notes" class="form-control" rows="3" placeholder="Optional notes about this item"></textarea>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="button" onclick="saveItem()" class="btn btn-primary">
                        <i class="material-icons mr-1" style="font-size: 16px;">save</i>
                        Save Item
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <style>
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
            border-radius: 4px;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-active {
            background-color: #d4edda;
            color: #155724;
        }

        .status-inactive {
            background-color: #f8d7da;
            color: #721c24;
        }

        .status-discontinued {
            background-color: #e2e3e5;
            color: #383d41;
        }

        /* Modal z-index fix */
        .modal,
        #addItemModal {
            z-index: 999999 !important;
            position: fixed !important;
        }

        .modal-backdrop,
        .modal-backdrop.fade,
        .modal-backdrop.show {
            z-index: 999998 !important;
            position: fixed !important;
        }

        .modal-dialog {
            z-index: 1000000 !important;
            position: relative !important;
        }

        .modal-content {
            z-index: 1000001 !important;
            position: relative !important;
        }

        /* Force modal to appear above everything */
        .modal.show,
        .modal.fade.show {
            display: block !important;
            z-index: 999999 !important;
            opacity: 1 !important;
        }

        /* Override any potential conflicts */
        .modal * {
            position: relative;
        }

        /* Ensure body doesn't scroll when modal is open */
        body.modal-open {
            overflow: hidden !important;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>

    <script>
        let itemsTable;

        $(document).ready(function() {
            initializeDataTable();

            // Entries per page change handler
            $('#entries-per-page').change(function() {
                itemsTable.page.len($(this).val()).draw();
            });
        });

        function initializeDataTable() {
            itemsTable = $('#items-table').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: '{{ route("warehouse.items.index") }}',
                    data: function(d) {
                        d.status = $('#filter-status').val();
                        d.category_id = $('#filter-category').val();
                    }
                },
                columns: [
                    {
                        data: null,
                        name: 'item_description',
                        render: function(data, type, row) {
                            return `
                                <div class="d-flex align-items-center">
                                    <div class="mr-3">
                                        <div class="bg-info text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                            <i class="material-icons" style="font-size: 20px;">inventory_2</i>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="font-weight-medium">${row.item_description}</div>
                                        <small class="text-muted">${row.item_code}</small>
                                    </div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'category_name',
                        name: 'category_name',
                        render: function(data) {
                            return `<span class="badge badge-secondary">${data}</span>`;
                        }
                    },
                    {
                        data: 'unit_of_measure',
                        name: 'unit_of_measure',
                        render: function(data) {
                            return `<span class="badge badge-light">${data}</span>`;
                        }
                    },
                    {
                        data: null,
                        name: 'stock_levels',
                        orderable: false,
                        render: function(data) {
                            return `
                                <div class="small">
                                    <div><strong>Min:</strong> ${data.min_stock_level || '-'}</div>
                                    <div><strong>Max:</strong> ${data.max_stock_level || '-'}</div>
                                    <div><strong>Reorder:</strong> ${data.reorder_point || '-'}</div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'status',
                        name: 'status',
                        render: function(data) {
                            const statusClasses = {
                                'active': 'status-active',
                                'inactive': 'status-inactive',
                                'discontinued': 'status-discontinued'
                            };
                            return `<span class="status-badge ${statusClasses[data] || 'status-inactive'}">${data.charAt(0).toUpperCase() + data.slice(1)}</span>`;
                        }
                    },
                    {
                        data: null,
                        name: 'created_at',
                        render: function(data) {
                            return `
                                <div>
                                    <div class="small">${data.created_at}</div>
                                    <div class="small text-muted">by ${data.created_by}</div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'actions',
                        orderable: false,
                        searchable: false,
                        render: function(data) {
                            return `
                                <div class="btn-group" role="group">
                                    <a href="{{ url('warehouse/items') }}/${data.id}" class="btn btn-sm btn-outline-primary" title="View">
                                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                                    </a>
                                    @can('warehouse.items.edit')
                                    <a href="{{ url('warehouse/items') }}/${data.id}/edit" class="btn btn-sm btn-outline-success" title="Edit">
                                        <i class="material-icons" style="font-size: 16px;">edit</i>
                                    </a>
                                    @endcan
                                    @can('warehouse.items.delete')
                                    <button onclick="deleteItem(${data.id})" class="btn btn-sm btn-outline-danger" title="Delete">
                                        <i class="material-icons" style="font-size: 16px;">delete</i>
                                    </button>
                                    @endcan
                                </div>
                            `;
                        }
                    }
                ],
                order: [[5, 'desc']],
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                language: {
                    processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
                    emptyTable: 'No items found',
                    zeroRecords: 'No matching items found'
                },
                drawCallback: function(settings) {
                    $('#entries-per-page').val(settings._iDisplayLength);
                }
            });
        }

        function applyFilters() {
            itemsTable.draw();
        }

        function clearFilters() {
            $('#filter-status, #filter-category').val('');
            itemsTable.draw();
        }

        function deleteItem(id) {
            if (confirm('Are you sure you want to delete this item?')) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: `{{ url('warehouse/items') }}/${id}`,
                    type: 'DELETE',
                    success: function(response) {
                        if (response.success) {
                            itemsTable.draw();
                            showAlert('Item deleted successfully', 'success');
                        } else {
                            showAlert(response.message, 'error');
                        }
                    },
                    error: function(xhr) {
                        showAlert('Error deleting item', 'error');
                    }
                });
            }
        }

        function showAddItemModal() {
            // Clear form and validation errors
            $('#addItemForm')[0].reset();
            clearValidationErrors();

            // Force very high z-index before showing
            $('#addItemModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Show modal
            $('#addItemModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Aggressively force z-index after modal is shown
            $('#addItemModal').on('shown.bs.modal', function() {
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 999998,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000,
                    'position': 'relative'
                });
                $('.modal-content').css({
                    'z-index': 1000001,
                    'position': 'relative'
                });

                // Move modal to body if it's not already there
                if (!$('#addItemModal').parent().is('body')) {
                    $('#addItemModal').appendTo('body');
                }
            });
        }

        function generateItemCode() {
            // Simple item code generation based on category and timestamp
            const categorySelect = $('#category_id');
            const selectedCategory = categorySelect.find(':selected').text();
            const categoryPrefix = selectedCategory.substring(0, 3).toUpperCase();
            const timestamp = Date.now().toString().slice(-6);
            const generatedCode = `${categoryPrefix}${timestamp}`;
            $('#item_code').val(generatedCode);
        }

        function saveItem() {
            const form = $('#addItemForm')[0];
            const formData = new FormData(form);

            // Clear previous validation errors
            clearValidationErrors();

            // Disable save button and show loading
            const saveBtn = $('button[onclick="saveItem()"]');
            const originalHtml = saveBtn.html();
            saveBtn.prop('disabled', true);
            saveBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Saving...');

            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            });

            $.ajax({
                url: '{{ route("warehouse.items.store") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        $('#addItemModal').modal('hide');
                        itemsTable.draw();
                        showAlert('Item added successfully!', 'success');
                    } else {
                        showAlert(response.message || 'Error adding item', 'error');
                    }
                },
                error: function(xhr) {
                    if (xhr.status === 422) {
                        // Validation errors
                        const errors = xhr.responseJSON.errors;
                        displayValidationErrors(errors);
                    } else {
                        showAlert('Error adding item. Please try again.', 'error');
                    }
                },
                complete: function() {
                    // Re-enable save button
                    saveBtn.prop('disabled', false);
                    saveBtn.html(originalHtml);
                }
            });
        }

        function clearValidationErrors() {
            $('.form-control').removeClass('is-invalid');
            $('.invalid-feedback').text('');
        }

        function displayValidationErrors(errors) {
            $.each(errors, function(field, messages) {
                const input = $(`[name="${field}"]`);
                input.addClass('is-invalid');
                input.siblings('.invalid-feedback').text(messages[0]);
            });
        }

        function showAlert(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            `;
            $('body').append(alert);
            setTimeout(() => $('.alert').alert('close'), 5000);
        }
    </script>
@endpush