<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Warehouse\DashboardController;
use App\Http\Controllers\Warehouse\ItemController;
use App\Http\Controllers\Warehouse\InventoryController;
use App\Http\Controllers\Warehouse\ItemCategoryController;
use App\Http\Controllers\Warehouse\IncomingOperationController;
use App\Http\Controllers\Warehouse\OutgoingTransactionController;
use App\Http\Controllers\Warehouse\ReturnTransactionController;
use App\Http\Controllers\Warehouse\ProjectController;
use App\Http\Controllers\Warehouse\ProjectDivisionController;
use App\Http\Controllers\Warehouse\SupplierController;
use App\Http\Controllers\Warehouse\LocationController;
use App\Http\Controllers\Warehouse\ReportController;
use App\Http\Controllers\Warehouse\AnalyticsController;
use App\Http\Controllers\Warehouse\SettingController;
use App\Http\Controllers\Warehouse\UserController;
use App\Http\Controllers\Warehouse\ActivityLogController;

/*
|--------------------------------------------------------------------------
| Warehouse Routes
|--------------------------------------------------------------------------
|
| Here are the routes for the warehouse management system. All routes
| are protected by the warehouse.access middleware and use role-based
| permissions for fine-grained access control.
|
*/

Route::prefix('warehouse')->name('warehouse.')->middleware(['auth', 'warehouse.access'])->group(function () {

    // Dashboard
    Route::get('/', [DashboardController::class, 'index'])->name('dashboard');
    Route::get('/api/quick-stats', [DashboardController::class, 'quickStats'])->name('api.quick-stats');

    // Items Management
    Route::prefix('items')->name('items.')->group(function () {
        Route::get('/', [ItemController::class, 'index'])->name('index');
        Route::get('/create', [ItemController::class, 'create'])->name('create');
        Route::post('/', [ItemController::class, 'store'])->name('store');
        Route::get('/{id}', [ItemController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ItemController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ItemController::class, 'update'])->name('update');
        Route::delete('/{id}', [ItemController::class, 'destroy'])->name('destroy');

        // Item specific actions
        Route::post('/{id}/generate-barcode', [ItemController::class, 'generateBarcode'])->name('generate-barcode');
        Route::post('/{id}/generate-qr-code', [ItemController::class, 'generateQrCode'])->name('generate-qr-code');
        Route::get('/low-stock/list', [ItemController::class, 'lowStock'])->name('low-stock');

        // Bulk operations
        Route::post('/bulk-import', [ItemController::class, 'bulkImport'])->name('bulk-import');
        Route::get('/export', [ItemController::class, 'export'])->name('export');

        // Categories
        Route::prefix('categories')->name('categories.')->group(function () {
            Route::get('/', [ItemCategoryController::class, 'index'])->name('index');
            Route::get('/create', [ItemCategoryController::class, 'create'])->name('create');
            Route::post('/', [ItemCategoryController::class, 'store'])->name('store');
            Route::get('/{id}', [ItemCategoryController::class, 'show'])->name('show');
            Route::get('/{id}/edit', [ItemCategoryController::class, 'edit'])->name('edit');
            Route::put('/{id}', [ItemCategoryController::class, 'update'])->name('update');
            Route::delete('/{id}', [ItemCategoryController::class, 'destroy'])->name('destroy');
        });
    });

    // Item Categories (standalone)
    Route::prefix('item-categories')->name('item-categories.')->group(function () {
        Route::get('/', [ItemCategoryController::class, 'index'])->name('index');
        Route::get('/create', [ItemCategoryController::class, 'create'])->name('create');
        Route::post('/', [ItemCategoryController::class, 'store'])->name('store');
        Route::get('/{id}', [ItemCategoryController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ItemCategoryController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ItemCategoryController::class, 'update'])->name('update');
        Route::delete('/{id}', [ItemCategoryController::class, 'destroy'])->name('destroy');
    });

    // Inventory Management
    Route::prefix('inventory')->name('inventory.')->group(function () {
        Route::get('/', [InventoryController::class, 'index'])->name('index');
        Route::get('/location/{location}', [InventoryController::class, 'byLocation'])->name('by-location');
        Route::get('/low-stock', [InventoryController::class, 'lowStock'])->name('low-stock');
        Route::get('/aging-report', [InventoryController::class, 'agingReport'])->name('aging-report');
        Route::get('/valuation', [InventoryController::class, 'valuation'])->name('valuation');

        // Inventory adjustments
        Route::get('/adjustments', [InventoryController::class, 'adjustments'])->name('adjustments');
        Route::get('/{itemId}/adjust', [InventoryController::class, 'adjust'])->name('adjust');
        Route::post('/{itemId}/adjust', [InventoryController::class, 'performAdjustment'])->name('perform-adjustment');

        // Inventory transfers
        Route::get('/transfers', [InventoryController::class, 'transfers'])->name('transfers');
        Route::get('/{itemId}/transfer', [InventoryController::class, 'transfer'])->name('transfer');
        Route::post('/{itemId}/transfer', [InventoryController::class, 'performTransfer'])->name('perform-transfer');

        // Inventory reservations
        Route::post('/{itemId}/reserve', [InventoryController::class, 'reserve'])->name('reserve');
        Route::post('/{itemId}/release-reserved', [InventoryController::class, 'releaseReserved'])->name('release-reserved');

        // Physical count
        Route::get('/physical-count', [InventoryController::class, 'physicalCount'])->name('physical-count');
        Route::post('/physical-count', [InventoryController::class, 'performPhysicalCount'])->name('perform-physical-count');

        // Stock movements
        Route::get('/{itemId}/stock-movements', [InventoryController::class, 'stockMovements'])->name('stock-movements');

        // Bulk operations
        Route::post('/check-bulk-availability', [InventoryController::class, 'checkBulkAvailability'])->name('check-bulk-availability');
    });

    // Enhanced Incoming Operations
    Route::prefix('incoming-operations')->name('incoming-operations.')->group(function () {
        Route::get('/', [IncomingOperationController::class, 'index'])->name('index');
        Route::get('/create', [IncomingOperationController::class, 'create'])->name('create');
        Route::post('/', [IncomingOperationController::class, 'store'])->name('store');
        Route::get('/{operation}', [IncomingOperationController::class, 'show'])->name('show');
        Route::get('/{operation}/edit', [IncomingOperationController::class, 'edit'])->name('edit');
        Route::put('/{operation}', [IncomingOperationController::class, 'update'])->name('update');
        Route::delete('/{operation}', [IncomingOperationController::class, 'destroy'])->name('destroy');

        // Quality Control & Inspection
        Route::post('/{operation}/inspect', [IncomingOperationController::class, 'inspect'])->name('inspect');
        Route::post('/{operation}/complete', [IncomingOperationController::class, 'complete'])->name('complete');

        // Printing & Documents
        Route::get('/{operation}/print', [IncomingOperationController::class, 'print'])->name('print');

        // API routes
        Route::get('/api/items', [IncomingOperationController::class, 'getItems'])->name('api.items');
        Route::get('/api/lpo-details', [IncomingOperationController::class, 'getLpoDetails'])->name('api.lpo-details');
        Route::get('/api/quality-checklists', [IncomingOperationController::class, 'getQualityChecklists'])->name('api.quality-checklists');
    });

    // Outgoing Transactions
    Route::prefix('outgoing')->name('outgoing.')->group(function () {
        Route::get('/', [OutgoingTransactionController::class, 'index'])->name('index');
        Route::get('/create', [OutgoingTransactionController::class, 'create'])->name('create');
        Route::post('/', [OutgoingTransactionController::class, 'store'])->name('store');
        Route::get('/{id}', [OutgoingTransactionController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [OutgoingTransactionController::class, 'edit'])->name('edit');
        Route::put('/{id}', [OutgoingTransactionController::class, 'update'])->name('update');
        Route::delete('/{id}', [OutgoingTransactionController::class, 'destroy'])->name('destroy');

        Route::get('/pending-approval/list', [OutgoingTransactionController::class, 'pendingApproval'])->name('pending-approval');
        Route::post('/{id}/approve', [OutgoingTransactionController::class, 'approve'])->name('approve');
        Route::post('/{id}/dispatch', [OutgoingTransactionController::class, 'dispatch'])->name('dispatch');
    });

    // Return Transactions
    Route::prefix('returns')->name('returns.')->group(function () {
        Route::get('/', [ReturnTransactionController::class, 'index'])->name('index');
        Route::get('/create', [ReturnTransactionController::class, 'create'])->name('create');
        Route::post('/', [ReturnTransactionController::class, 'store'])->name('store');
        Route::get('/{id}', [ReturnTransactionController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ReturnTransactionController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ReturnTransactionController::class, 'update'])->name('update');
        Route::delete('/{id}', [ReturnTransactionController::class, 'destroy'])->name('destroy');

        Route::post('/{id}/process', [ReturnTransactionController::class, 'process'])->name('process');
    });

    // Projects
    Route::prefix('projects')->name('projects.')->group(function () {
        Route::get('/', [ProjectController::class, 'index'])->name('index');
        Route::get('/create', [ProjectController::class, 'create'])->name('create');
        Route::post('/', [ProjectController::class, 'store'])->name('store');
        Route::get('/{id}', [ProjectController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ProjectController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ProjectController::class, 'update'])->name('update');
        Route::delete('/{id}', [ProjectController::class, 'destroy'])->name('destroy');
    });

    // Project Divisions
    Route::prefix('project-divisions')->name('project-divisions.')->group(function () {
        Route::get('/', [ProjectDivisionController::class, 'index'])->name('index');
        Route::get('/create', [ProjectDivisionController::class, 'create'])->name('create');
        Route::post('/', [ProjectDivisionController::class, 'store'])->name('store');
        Route::get('/{id}', [ProjectDivisionController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [ProjectDivisionController::class, 'edit'])->name('edit');
        Route::put('/{id}', [ProjectDivisionController::class, 'update'])->name('update');
        Route::delete('/{id}', [ProjectDivisionController::class, 'destroy'])->name('destroy');
    });

    // Suppliers
    Route::prefix('suppliers')->name('suppliers.')->group(function () {
        Route::get('/', [SupplierController::class, 'index'])->name('index');
        Route::get('/data', [SupplierController::class, 'data'])->name('data');
        Route::get('/create', [SupplierController::class, 'create'])->name('create');
        Route::post('/', [SupplierController::class, 'store'])->name('store');
        Route::get('/{id}', [SupplierController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [SupplierController::class, 'edit'])->name('edit');
        Route::put('/{id}', [SupplierController::class, 'update'])->name('update');
        Route::delete('/{id}', [SupplierController::class, 'destroy'])->name('destroy');

        // Additional supplier functionality
        Route::post('/import', [SupplierController::class, 'import'])->name('import');
        Route::get('/export', [SupplierController::class, 'export'])->name('export');
        Route::post('/{id}/update-rating', [SupplierController::class, 'updateRating'])->name('updateRating');
        Route::get('/{id}/performance', [SupplierController::class, 'performance'])->name('performance');
        Route::get('/filter-options', [SupplierController::class, 'filterOptions'])->name('filterOptions');
        Route::get('/dropdown', [SupplierController::class, 'dropdown'])->name('dropdown');
    });

    // Locations
    Route::prefix('locations')->name('locations.')->group(function () {
        Route::get('/', [LocationController::class, 'index'])->name('index');
        Route::get('/create', [LocationController::class, 'create'])->name('create');
        Route::post('/', [LocationController::class, 'store'])->name('store');
        Route::get('/{id}', [LocationController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [LocationController::class, 'edit'])->name('edit');
        Route::put('/{id}', [LocationController::class, 'update'])->name('update');
        Route::delete('/{id}', [LocationController::class, 'destroy'])->name('destroy');
    });

    // Reports
    Route::prefix('reports')->name('reports.')->group(function () {
        Route::get('/', [ReportController::class, 'index'])->name('index');
        Route::get('/inventory', [ReportController::class, 'inventory'])->name('inventory');
        Route::get('/stock-movements', [ReportController::class, 'stockMovements'])->name('stock-movements');
        Route::get('/transactions', [ReportController::class, 'transactions'])->name('transactions');
        Route::get('/aging', [ReportController::class, 'aging'])->name('aging');
        Route::get('/low-stock', [ReportController::class, 'lowStock'])->name('low-stock');
        Route::get('/supplier-performance', [ReportController::class, 'supplierPerformance'])->name('supplier-performance');
    });

    // Analytics
    Route::prefix('analytics')->name('analytics.')->group(function () {
        Route::get('/', [AnalyticsController::class, 'index'])->name('index');
        Route::get('/inventory-turnover', [AnalyticsController::class, 'inventoryTurnover'])->name('inventory-turnover');
        Route::get('/stock-forecast', [AnalyticsController::class, 'stockForecast'])->name('stock-forecast');
        Route::get('/performance-metrics', [AnalyticsController::class, 'performanceMetrics'])->name('performance-metrics');
    });

    // Settings
    Route::prefix('settings')->name('settings.')->group(function () {
        Route::get('/', [SettingController::class, 'index'])->name('index');
        Route::put('/general', [SettingController::class, 'updateGeneral'])->name('update-general');
        Route::put('/notifications', [SettingController::class, 'updateNotifications'])->name('update-notifications');
        Route::put('/workflows', [SettingController::class, 'updateWorkflows'])->name('update-workflows');
    });

    // User Management
    Route::prefix('users')->name('users.')->group(function () {
        Route::get('/', [UserController::class, 'index'])->name('index');
        Route::get('/create', [UserController::class, 'create'])->name('create');
        Route::post('/', [UserController::class, 'store'])->name('store');
        Route::get('/{id}', [UserController::class, 'show'])->name('show');
        Route::get('/{id}/edit', [UserController::class, 'edit'])->name('edit');
        Route::put('/{id}', [UserController::class, 'update'])->name('update');
        Route::delete('/{id}', [UserController::class, 'destroy'])->name('destroy');
        Route::post('/{id}/assign-role', [UserController::class, 'assignRole'])->name('assign-role');
    });

    // Activity Log
    Route::prefix('activity-log')->name('activity-log.')->group(function () {
        Route::get('/', [ActivityLogController::class, 'index'])->name('index');
        Route::get('/{id}', [ActivityLogController::class, 'show'])->name('show');
        Route::delete('/cleanup', [ActivityLogController::class, 'cleanup'])->name('cleanup');
    });
});